<?php

namespace Paysera\Hook;

use Paysera\Entity\CheckoutData;
use Paysera\Exception\PaymentListJavascriptException;
use Paysera\Helper\AssetsHelper;
use Paysera\Provider\ClientDataProvider;
use Paysera\Service\PayseraSmarty;
use SmartyException;

class PaymentListJavascriptHook
{
    const PAYMENT_LIST_JAVASCRIPT_TEMPLATE = 'paymentListJavascript.tpl';

    private $clientDataProvider;
    private $smarty;
    private $assetsHelper;

    public function __construct()
    {
        $this->clientDataProvider = new ClientDataProvider();
        $this->smarty = (new PayseraSmarty())->buildSmarty();
        $this->assetsHelper = new AssetsHelper();
    }

    /**
     * @param CheckoutData $checkoutData
     * @return string
     * @throws SmartyException|PaymentListJavascriptException
     */
    public function builPaymentListJavascriptResponse(CheckoutData $checkoutData)
    {
        $this->smarty->assign('amount', $this->resolveCheckoutAmount($checkoutData));
        $this->smarty->assign('currency', $this->resolveCheckoutCurrency($checkoutData));
        $this->smarty->assign('webRootDir', $this->assetsHelper->getWebRootDir());

        return $this->smarty->fetch(self::PAYMENT_LIST_JAVASCRIPT_TEMPLATE);
    }

    /**
     * @param CheckoutData $checkoutData
     * @return float|null
     * @throws PaymentListJavascriptException
     */
    private function resolveCheckoutAmount(CheckoutData $checkoutData)
    {
        switch (true) {
            case $checkoutData->getRawTotal() !== null: {
                return $checkoutData->getRawTotal();
            }
            case $checkoutData->getPriceTotal() !== null: {
                return $checkoutData->getPriceTotal()->format();
            }
            case $checkoutData->getAmount() !== null: {
                return $checkoutData->getAmount();
            }
        }

        throw new PaymentListJavascriptException('Checkout amount could not be resolved');
    }

    /**
     * @param CheckoutData $checkoutData
     * @return string|null
     * @throws PaymentListJavascriptException
     */
    private function resolveCheckoutCurrency(CheckoutData $checkoutData)
    {
        switch (true) {
            case $checkoutData->getCurrency() !== null: {
                return $checkoutData->getCurrency();
            }
            case $checkoutData->getPriceTotal() !== null: {
                $currency = $checkoutData->getPriceTotal()->getCurrency();

                if (isset($currency['code'])) {
                    return (string) $currency['code'];
                }
            }
            case $this->clientDataProvider->getClientCurrency() !== null: {
                return $this->clientDataProvider->getClientCurrency();
            }
        }

        throw new PaymentListJavascriptException('Checkout currency could not be resolved');
    }
}
