<?php

namespace Paysera\Controller;

use Paysera\Helper\LanguageHelper;
use Paysera\Helper\PaymentHelper;
use Paysera\Normalizer\CheckoutDataNormalizer;
use Paysera\Service\PayseraSmarty;
use Paysera\Provider\PluginSettingsProvider;
use Paysera\Exception\PaymentListRequestValidatorException;
use Paysera\Validator\PaymentListRequestValidator;
use SmartyException;
use WebToPayException;
use App;

class PaymentListController
{
    const CHECKOUT_FORM_TEMPLATE = 'paymentList.tpl';

    private $payseraSmarty;
    private $pluginSettings;
    private $languageHelper;
    private $paymentHelper;
    private $checkoutDataNormalizer;
    private $paymentListRequestValidator;

    public function __construct()
    {
        $this->payseraSmarty = (new PayseraSmarty())->buildSmarty();
        $this->pluginSettings = (new PluginSettingsProvider())->getPluginSettings();
        $this->languageHelper = new LanguageHelper();
        $this->paymentHelper = new PaymentHelper();
        $this->checkoutDataNormalizer = new CheckoutDataNormalizer();
        $this->paymentListRequestValidator = new PaymentListRequestValidator();
    }

    /**
     * @param array|null $whmcsLang
     * @return string
     * @throws PaymentListRequestValidatorException|WebToPayException|SmartyException
     */
    public function buildPaymentListResponse($whmcsLang)
    {
        if (!$this->pluginSettings->isPaymentListEnabled()) {
            return '';
        }

        $this->paymentListRequestValidator->validateRequestData(
            $this->getRequestAmount(),
            $this->getRequestCurrency(),
            $this->getRequestCountry()
        );

        $requestData = $this->checkoutDataNormalizer->mapToEntity($this->getRequestMap());

        $paymentMethodList = $this->paymentHelper->getPaymentMethodList(
            $this->pluginSettings,
            $this->languageHelper->getClientLanguage($whmcsLang),
            $requestData->getAmount(),
            $requestData->getCurrency()
        );

        $activeCountry = $this->paymentHelper->getActiveCountry(
            $this->pluginSettings,
            $paymentMethodList,
            strtolower($requestData->getBillingCountry())
        );

        $this->payseraSmarty->assign('description', $this->pluginSettings->getDescription());
        $this->payseraSmarty->assign('gridViewEnabled', $this->pluginSettings->isGridViewEnabled());
        $this->payseraSmarty->assign('buyerConsentEnabled', $this->pluginSettings->isBuyerConsentEnabled());
        $this->payseraSmarty->assign('enabledCountries', $this->pluginSettings->getEnabledCountries());
        $this->payseraSmarty->assign('paymentMethodList', $paymentMethodList);
        $this->payseraSmarty->assign('activeCountry', $activeCountry);

        return $this->payseraSmarty->display(self::CHECKOUT_FORM_TEMPLATE);
    }

    private function getRequestAmount()
    {
        return App::getFromRequest('paysera_amount');
    }

    private function getRequestCurrency()
    {
        return App::getFromRequest('paysera_currency');
    }

    private function getRequestCountry()
    {
        return App::getFromRequest('paysera_country');
    }

    private function getRequestMap()
    {
        return [
            'amount' => $this->getRequestAmount(),
            'currency' =>  $this->getRequestCurrency(),
            'country' => $this->getRequestCountry(),
        ];
    }
}
