<?php

namespace Paysera\Controller;

use Paysera\Entity\PluginSettings;
use Paysera\Exception\CallbackControllerException;
use Paysera\Provider\PluginSettingsProvider;
use WebToPay;
use WHMCS\Billing\Invoice;
use WebToPayException;

class CallbackController
{
    private $pluginSettings;
    private $gatewayParameters;

    public function __construct()
    {
        $this->pluginSettings = (new PluginSettingsProvider())->getPluginSettings();
        $this->gatewayParameters = getGatewayVariables(PluginSettings::PLUGIN_NAME);
    }

    /**
     * @param array $request
     * @return string
     * @throws CallbackControllerException|WebToPayException
     */
    public function acceptPayment($request)
    {
        $response = WebToPay::validateAndParseData(
            $request,
            $this->pluginSettings->getProjectId(),
            $this->pluginSettings->getProjectPassword()
        );

        $invoiceId = $response['orderid'];
        $requestId = $response['requestid'];
        $status = $response['status'];

        $this->validateRequest($invoiceId, $requestId, $response['type']);

        if ($status === '1') {
            /** @var Invoice $invoice */
            $invoice = Invoice::findOrFail($invoiceId);

            $invoiceTotal = $this->getInvoiceTotal($invoice);
            $invoiceCurrency = $this->getInvoiceCurrency($invoice);

            $this->validatePayment(
                $response['amount'],
                $response['currency'],
                $response['payamount'],
                $response['paycurrency'],
                $this->convertAmountToStringCents($invoiceTotal),
                $invoiceCurrency
            );

            $addInvoicePayment = addInvoicePayment(
                $invoiceId,
                $requestId,
                $invoice->getAttributeValue('total'),
                null,
                PluginSettings::PLUGIN_NAME
            );

            logTransaction(PluginSettings::PLUGIN_NAME, $response, 'Payment completed');

            return $addInvoicePayment ? 'OK invoice payment added' : 'ERROR an error occurred while adding payment';
        }

        return sprintf('OK no action taken because response status %d', $status);
    }

    /**
     * @param Invoice $invoice
     * @return float
     */
    private function getInvoiceTotal(Invoice $invoice)
    {
        $convertTo = !isset($this->gatewayParameters['convertto']) ?: $this->gatewayParameters['convertto'];
        $invoiceTotal = $invoice->getAttributeValue('total');

        if (!empty($convertTo)) {
            $clientCurrency = getCurrency($invoice->getAttributeValue('userid'))['id'];

            return convertCurrency($invoiceTotal, $clientCurrency, $convertTo);
        }

        return $invoiceTotal;
    }

    /**
     * @param Invoice $invoice
     * @return string
     */
    private function getInvoiceCurrency(Invoice $invoice)
    {
        $convertTo = !isset($this->gatewayParameters['convertto']) ?: $this->gatewayParameters['convertto'];

        if (!empty($convertTo)) {
            return getCurrency('', $convertTo)['code'];
        }

        return getCurrency($invoice->getAttributeValue('userid'))['code'];
    }

    private function convertAmountToStringCents($amount)
    {
        return (string) (round($amount, 2) * 100);
    }

    /**
     * @param string $amount
     * @param string $amountCurrency
     * @param string $payAmount
     * @param string $payCurrency
     * @param string $invoiceAmount
     * @param string $clientCurrency
     * @throws CallbackControllerException
     */
    private function validatePayment(
        $amount,
        $amountCurrency,
        $payAmount,
        $payCurrency,
        $invoiceAmount,
        $clientCurrency
    ) {
        if ($amount !== $invoiceAmount || $amountCurrency !== $clientCurrency) {
            throw new CallbackControllerException(
                sprintf(
                    'Payment doesnt match invoice: \'%s %s\', expected \'%s %s\'',
                    $amount,
                    $amountCurrency,
                    $invoiceAmount,
                    $clientCurrency
                )
            );
        }

        if ($payAmount > 0  && ($payAmount !== $invoiceAmount || $payCurrency !== $clientCurrency)) {
            throw new CallbackControllerException(
                sprintf(
                    'Converted Paysera payment doesnt match invoice: \'%s %s\', expected \'%s %s\'',
                    $payAmount,
                    $payCurrency,
                    $invoiceAmount,
                    $clientCurrency
                )
            );
        }
    }

    /**
     * @param string $invoiceId
     * @param string $requestId
     * @param string $requestType
     * @throws CallbackControllerException
     */
    private function validateRequest($invoiceId, $requestId, $requestType)
    {
        checkCbInvoiceID($invoiceId, PluginSettings::PLUGIN_NAME);
        checkCbTransID($requestId);

        if ($requestType !== 'macro') {
            throw new CallbackControllerException('Only macro payments are accepted');
        }
    }
}
