<?php

declare(strict_types=1);

namespace SwagPayseraCheckout\Service;

use SwagPayseraCheckout\Entity\PaymentRequest;
use SwagPayseraCheckout\Exception\PaymentRequestValidationException;
use SwagPayseraCheckout\Exception\ProjectSettingsException;
use SwagPayseraCheckout\Normalizer\PaymentRequestNormalizer;
use SwagPayseraCheckout\Provider\PluginSettingsProvider;
use SwagPayseraCheckout\Service\WebToPay\WebToPayLoader;
use SwagPayseraCheckout\Validator\PaymentRequestValidator;
use WebToPay;
use WebToPayException;

class PaymentRequestResolver
{
    private $pluginSettingsProvider;
    private $paymentRequestNormalizer;
    private $paymentRequestValidator;

    public function __construct(
        PluginSettingsProvider $pluginSettingsProvider,
        PaymentRequestNormalizer $paymentRequestNormalizer,
        PaymentRequestValidator $paymentRequestValidator
    ) {
        $this->pluginSettingsProvider = $pluginSettingsProvider;
        $this->paymentRequestNormalizer = $paymentRequestNormalizer;
        $this->paymentRequestValidator = $paymentRequestValidator;
    }

    /**
     * @param array $request
     * @return PaymentRequest
     * @throws PaymentRequestValidationException|WebToPayException|ProjectSettingsException
     */
    public function resolvePaymentRequest(array $request): PaymentRequest
    {
        WebToPayLoader::loadWebToPay();

        $pluginSettings = $this->pluginSettingsProvider->getProjectSettingsByShopId(
            $this->resolveAndValidateShopId($request)
        );

        $paymentRequestData = WebToPay::validateAndParseData(
            $request,
            $pluginSettings->getProjectId(),
            $pluginSettings->getProjectSign()
        );

        $this->paymentRequestValidator->validatePaymentRequest($paymentRequestData);

        return $this->paymentRequestNormalizer->mapToWebToPayRequest($paymentRequestData);
    }

    /**
     * @param array $request
     * @return int|null
     * @throws PaymentRequestValidationException
     */
    private function resolveAndValidateShopId(array $request): ?int
    {
        if (!isset($request['data'])) {
            throw new PaymentRequestValidationException('Missing data parameter');
        }

        $decoded = base64_decode($request['data'], true);

        if ($decoded === false) {
            throw new PaymentRequestValidationException('Data cannot be decoded');
        }

        $parsed = [];

        parse_str($decoded, $parsed);

        $shopId = $parsed[PaymentRequest::SHOP_ID_KEY] ?? null;

        if ($shopId === null) {
            return null;
        }

        if (!filter_var($shopId, FILTER_VALIDATE_INT)) {
            throw new PaymentRequestValidationException(
                sprintf(
                    'Given %s parameter \'%s\' is not valid',
                    PaymentRequest::SHOP_ID_KEY,
                    htmlspecialchars($shopId)
                )
            );
        }

        $shopId = (int) $shopId;

        return $shopId === 0 ? null : $shopId;
    }
}
