<?php

declare(strict_types=1);

namespace SwagPayseraCheckout\Validator;

use Shopware\Models\Order\Order;
use SwagPayseraCheckout\Entity\PaymentRequest;
use SwagPayseraCheckout\Exception\PaymentValidatorException;
use SwagPayseraCheckout\Service\AmountFormatter;

class PaymentValidator
{
    private $amountFormatter;

    public function __construct(AmountFormatter $amountFormatter)
    {
        $this->amountFormatter = $amountFormatter;
    }

    /**
     * @param PaymentRequest $paymentRequest
     * @param Order $order
     * @throws PaymentValidatorException
     */
    public function validatePayment(PaymentRequest $paymentRequest, Order $order): void
    {
        $payAmount = $paymentRequest->getPayAmount() ?? 0;
        $payCurrency = $paymentRequest->getPayCurrency() ?? 'empty';

        $orderAmountInCents = $this->amountFormatter->formatAmountInCents($order->getInvoiceAmount());
        $orderCurrency = $order->getCurrency();

        if (
            $paymentRequest->getAmount() !== $orderAmountInCents
            || $paymentRequest->getCurrency() !== $orderCurrency
        ) {
            throw new PaymentValidatorException(
                sprintf(
                    'Payment doesnt match invoice: \'%s %s\', expected \'%s %s\'',
                    $paymentRequest->getAmount(),
                    $paymentRequest->getCurrency(),
                    $orderAmountInCents,
                    $orderCurrency
                )
            );
        }

        if ($payAmount > 0  && ($payAmount !== $orderAmountInCents || $payCurrency !== $orderCurrency)) {
            throw new PaymentValidatorException(
                sprintf(
                    'Converted Paysera payment doesnt match invoice: \'%s %s\', expected \'%s %s\'',
                    $payAmount,
                    $payCurrency,
                    $orderAmountInCents,
                    $orderCurrency
                )
            );
        }
    }
}
