<?php

declare(strict_types=1);

use Shopware\Components\CSRFWhitelistAware;
use Shopware\Components\Logger;
use Shopware\Models\Order\Status;
use SwagPayseraCheckout\Entity\PaymentRequest;
use SwagPayseraCheckout\Exception\PaymentRequestValidationException;
use SwagPayseraCheckout\Exception\PayseraCheckoutException;
use SwagPayseraCheckout\Provider\OrderProvider;
use SwagPayseraCheckout\Service\PaymentHashHelper;
use SwagPayseraCheckout\Service\PaymentRequestBuilder;
use SwagPayseraCheckout\Service\PaymentRequestResolver;
use SwagPayseraCheckout\Validator\PaymentValidator;

class Shopware_Controllers_Frontend_PayseraPayment extends Shopware_Controllers_Frontend_Payment implements CSRFWhitelistAware
{
    /**
     * @var PaymentHashHelper
     */
    private $paymentHashHelper;

    /**
     * @var PaymentRequestBuilder
     */
    private $paymentRequestBuilder;

    /**
     * @var PaymentRequestResolver
     */
    private $paymentRequestResolver;

    /**
     * @var PaymentValidator
     */
    private $paymentValidator;

    /**
     * @var OrderProvider
     */
    private $orderProvider;

    /**
     * @var Logger
     */
    private $logger;

    public function preDispatch(): void
    {
        $this->paymentHashHelper = $this->get('swag_paysera_checkout.service.payment_hash_helper');
        $this->paymentRequestBuilder = $this->get('swag_paysera_checkout.service.payment_request_builder');
        $this->paymentRequestResolver = $this->get('swag_paysera_checkout.service.payment_request_resolver');
        $this->paymentValidator = $this->get('swag_paysera_checkout.validator.payment_validator');
        $this->orderProvider = $this->get('swag_paysera_checkout.provider.order_provider');
        $this->logger = $this->get('pluginlogger');

        $this->get('template')->addTemplateDir(__DIR__ . '/../../Resources/views/');
    }

    public function indexAction(): void
    {
        $this->forward('direct');
    }

    public function directAction(): void
    {
        try {
            $user = $this->getUser() ?? $this->getUserSessionFallback();
            $shopId = $this->getShopId();
            $router = $this->Front()->Router();

            $paymentHash = $this->paymentHashHelper->getPaymentHash($user, $this->getAmount());
            $uniquePaymentId = $this->createPaymentUniqueId();

            $orderNumber = $this->saveOrder(
                $paymentHash,
                $uniquePaymentId,
                Status::PAYMENT_STATE_THE_PAYMENT_HAS_BEEN_ORDERED
            );

            if ($orderNumber === false) {
                $this->logger->warning(
                    'Order creation failed',
                    [
                        'paymentHash' => $paymentHash,
                        'uniquePaymentId' => $uniquePaymentId,
                    ]
                );

                $this->redirect(['controller' => 'checkout', 'action' => 'confirm']);
            } else {
                $paymentRequestUrl = $this->paymentRequestBuilder->getPaymentRequestUrl(
                    (string) $orderNumber,
                    $user,
                    $router->assemble(['action' => 'return', 'forceSecure' => true]),
                    $router->assemble(['action' => 'cancel', 'forceSecure' => true]),
                    $router->assemble(['action' => 'notify', 'forceSecure' => true]),
                    $this->getAmount(),
                    $this->getCurrencyShortName(),
                    $paymentHash,
                    $uniquePaymentId,
                    $shopId
                );

                $this->redirect($paymentRequestUrl);
            }
        } catch (WebToPayException $payseraCheckoutException) {
            $this->logger->debug($payseraCheckoutException->getMessage(), [$payseraCheckoutException]);
        }
    }

    public function returnAction(): void
    {
        try {
            $paymentRequest = $this->getPaymentRequest();

            $this->savePaymentStatus(
                $paymentRequest->getPaymentHash(),
                $paymentRequest->getUniquePaymentId(),
                Status::PAYMENT_STATE_OPEN
            );

            $this->redirect(['controller' => 'checkout', 'action' => 'finish']);
        } catch (PayseraCheckoutException|WebToPayException $exception) {
            $this->logger->debug($exception->getMessage(), [$exception]);
        }
    }

    public function notifyAction()
    {
        try {
            $this->container->get('front')->Plugins()->ViewRenderer()->setNoRender();

            $paymentRequest = $this->getPaymentRequest();
            $order = $this->orderProvider->getOrderByNumber($paymentRequest->getOrderId());

            $this->paymentValidator->validatePayment($paymentRequest, $order);

            $response = null;

            if ($paymentRequest->getType() === 'macro') {
                if ($paymentRequest->getStatus() === 1) {
                    $this->savePaymentStatus(
                        $paymentRequest->getPaymentHash(),
                        $paymentRequest->getUniquePaymentId(),
                        Status::PAYMENT_STATE_COMPLETELY_PAID
                    );

                    $response = 'OK payment was successful';
                } else {
                    $response = 'OK nothing was done because payment status is not equal to 1';
                }
            } else {
                $response = 'ERROR only macro payment types are accepted';
            }

            $this->Response()
                ->setBody($response)
            ;
        } catch (PayseraCheckoutException|WebToPayException $exception) {
            $this->logger->debug($exception->getMessage(), [$exception]);

            echo 'ERROR: ' . $exception->getMessage();
        }
    }

    public function cancelAction()
    {
        $this->redirect(['controller' => 'checkout', 'action' => 'cart']);
    }

    public function getWhitelistedCSRFActions(): array
    {
        return ['notify'];
    }

    private function getShopId(): ?int
    {
        if ($this->getUser() === null) {
            return null;
        }

        $shopId = $this->getBasket()['user']['subshopID'] ?? null;

        if ($shopId === null) {
            return null;
        }

        return (int) $shopId;
    }

    /**
     * @return PaymentRequest
     * @throws WebToPayException
     * @throws PaymentRequestValidationException
     */
    private function getPaymentRequest(): PaymentRequest
    {
        return $this->paymentRequestResolver->resolvePaymentRequest(
            $this->request->query->all()
        );
    }

    private function getUserSessionFallback(): array
    {
        $fallback = [];
        $fallback['additional']['user']['sessionID'] = $this->createPaymentUniqueId();

        return $fallback;
    }
}
