<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace Paysera\Scoped\Symfony\Component\Validator\Constraints;

use Paysera\Scoped\Symfony\Component\Validator\Constraint;
use Paysera\Scoped\Symfony\Component\Validator\ConstraintValidator;
use Paysera\Scoped\Symfony\Component\Validator\Exception\UnexpectedTypeException;
use Paysera\Scoped\Symfony\Component\Validator\Exception\UnexpectedValueException;
/**
 * @author Bernhard Schussek <bschussek@gmail.com>
 */
class TimeValidator extends ConstraintValidator
{
    public const PATTERN = '/^(\\d{2}):(\\d{2}):(\\d{2})$/';
    /**
     * Checks whether a time is valid.
     *
     * @internal
     */
    public static function checkTime(int $hour, int $minute, float $second) : bool
    {
        return $hour >= 0 && $hour < 24 && $minute >= 0 && $minute < 60 && $second >= 0 && $second < 60;
    }
    /**
     * {@inheritdoc}
     */
    public function validate($value, Constraint $constraint)
    {
        if (!$constraint instanceof Time) {
            throw new UnexpectedTypeException($constraint, Time::class);
        }
        if (null === $value || '' === $value) {
            return;
        }
        if ($value instanceof \DateTimeInterface) {
            @\trigger_error(\sprintf('Validating a \\DateTimeInterface with "%s" is deprecated since version 4.2. Use "%s" instead or remove the constraint if the underlying model is already type hinted to \\DateTimeInterface.', Time::class, Type::class), \E_USER_DEPRECATED);
            return;
        }
        if (!\is_scalar($value) && !(\is_object($value) && \method_exists($value, '__toString'))) {
            throw new UnexpectedValueException($value, 'string');
        }
        $value = (string) $value;
        if (!\preg_match(static::PATTERN, $value, $matches)) {
            $this->context->buildViolation($constraint->message)->setParameter('{{ value }}', $this->formatValue($value))->setCode(Time::INVALID_FORMAT_ERROR)->addViolation();
            return;
        }
        if (!self::checkTime($matches[1], $matches[2], $matches[3])) {
            $this->context->buildViolation($constraint->message)->setParameter('{{ value }}', $this->formatValue($value))->setCode(Time::INVALID_TIME_ERROR)->addViolation();
        }
    }
}
