<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace Paysera\Scoped\Symfony\Component\Validator\Constraints;

use Paysera\Scoped\Symfony\Component\Validator\Constraint;
use Paysera\Scoped\Symfony\Component\Validator\ConstraintValidator;
use Paysera\Scoped\Symfony\Component\Validator\Exception\UnexpectedTypeException;
use Paysera\Scoped\Symfony\Component\Validator\Exception\UnexpectedValueException;
/**
 * @author Bernhard Schussek <bschussek@gmail.com>
 */
class LengthValidator extends ConstraintValidator
{
    /**
     * {@inheritdoc}
     */
    public function validate($value, Constraint $constraint)
    {
        if (!$constraint instanceof Length) {
            throw new UnexpectedTypeException($constraint, Length::class);
        }
        if (null !== $constraint->min && null === $constraint->allowEmptyString) {
            @\trigger_error(\sprintf('Using the "%s" constraint with the "min" option without setting the "allowEmptyString" one is deprecated and defaults to true. In 5.0, it will become optional and default to false.', Length::class), \E_USER_DEPRECATED);
        }
        if (null === $value || '' === $value && ($constraint->allowEmptyString ?? \true)) {
            return;
        }
        if (!\is_scalar($value) && !(\is_object($value) && \method_exists($value, '__toString'))) {
            throw new UnexpectedValueException($value, 'string');
        }
        $stringValue = (string) $value;
        if (null !== $constraint->normalizer) {
            $stringValue = ($constraint->normalizer)($stringValue);
        }
        try {
            $invalidCharset = !@\mb_check_encoding($stringValue, $constraint->charset);
        } catch (\ValueError $e) {
            if (!\str_starts_with($e->getMessage(), 'mb_check_encoding(): Argument #2 ($encoding) must be a valid encoding')) {
                throw $e;
            }
            $invalidCharset = \true;
        }
        if ($invalidCharset) {
            $this->context->buildViolation($constraint->charsetMessage)->setParameter('{{ value }}', $this->formatValue($stringValue))->setParameter('{{ charset }}', $constraint->charset)->setInvalidValue($value)->setCode(Length::INVALID_CHARACTERS_ERROR)->addViolation();
            return;
        }
        $length = \mb_strlen($stringValue, $constraint->charset);
        if (null !== $constraint->max && $length > $constraint->max) {
            $this->context->buildViolation($constraint->min == $constraint->max ? $constraint->exactMessage : $constraint->maxMessage)->setParameter('{{ value }}', $this->formatValue($stringValue))->setParameter('{{ limit }}', $constraint->max)->setInvalidValue($value)->setPlural((int) $constraint->max)->setCode(Length::TOO_LONG_ERROR)->addViolation();
            return;
        }
        if (null !== $constraint->min && $length < $constraint->min) {
            $this->context->buildViolation($constraint->min == $constraint->max ? $constraint->exactMessage : $constraint->minMessage)->setParameter('{{ value }}', $this->formatValue($stringValue))->setParameter('{{ limit }}', $constraint->min)->setInvalidValue($value)->setPlural((int) $constraint->min)->setCode(Length::TOO_SHORT_ERROR)->addViolation();
        }
    }
}
