<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace Paysera\Scoped\Symfony\Component\Validator\Constraints;

use Paysera\Scoped\Symfony\Component\Intl\Languages;
use Paysera\Scoped\Symfony\Component\Validator\Constraint;
use Paysera\Scoped\Symfony\Component\Validator\ConstraintValidator;
use Paysera\Scoped\Symfony\Component\Validator\Exception\LogicException;
use Paysera\Scoped\Symfony\Component\Validator\Exception\UnexpectedTypeException;
use Paysera\Scoped\Symfony\Component\Validator\Exception\UnexpectedValueException;
/**
 * Validates whether a value is a valid language code.
 *
 * @author Bernhard Schussek <bschussek@gmail.com>
 */
class LanguageValidator extends ConstraintValidator
{
    /**
     * {@inheritdoc}
     */
    public function validate($value, Constraint $constraint)
    {
        if (!$constraint instanceof Language) {
            throw new UnexpectedTypeException($constraint, Language::class);
        }
        if (null === $value || '' === $value) {
            return;
        }
        if (!\is_scalar($value) && !(\is_object($value) && \method_exists($value, '__toString'))) {
            throw new UnexpectedValueException($value, 'string');
        }
        if (!\class_exists(Languages::class)) {
            throw new LogicException('The Intl component is required to use the Language constraint. Try running "composer require symfony/intl".');
        }
        $value = (string) $value;
        if (!Languages::exists($value)) {
            $this->context->buildViolation($constraint->message)->setParameter('{{ value }}', $this->formatValue($value))->setCode(Language::NO_SUCH_LANGUAGE_ERROR)->addViolation();
        }
    }
}
