<?php
/**
 * 2018 Paysera
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/afl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * @author Paysera <plugins@paysera.com>
 * @copyright 2018 Paysera
 * @license http://opensource.org/licenses/afl-3.0.php Academic Free License (AFL 3.0)
 *  International Registered Trademark & Property of Paysera
 */
namespace Paysera\Form;

if (!defined('_PS_VERSION_')) {
    exit;
}

use Paysera\Entity\FormResult;
use Paysera\Helper\FormHelperInterface;

class StatusSettingsForm implements AdminFormInterface
{
    /**
     * @var FormHelperInterface
     */
    private $formHelper;
    /**
     * @var \Module
     */
    private $module;
    /**
     * @var string
     */
    private $formId;

    public function __construct(FormHelperInterface $formHelper, \Module $module)
    {
        $this->formHelper = $formHelper;
        $this->module = $module;
        $this->formId = 'StatusSettingsForm';
    }

    /**
     * @return FormResult
     */
    public function validate()
    {
        return new FormResult(true, '');
    }

    /**
     * @return FormResult
     */
    public function update()
    {
        \Configuration::updateValue(
            'PAYSERA_ORDER_STATUS_NEW',
            \Tools::getValue('PAYSERA_ORDER_STATUS_NEW')
        );
        \Configuration::updateValue(
            'PAYSERA_ORDER_STATUS_PAID',
            \Tools::getValue('PAYSERA_ORDER_STATUS_PAID')
        );
        \Configuration::updateValue(
            'PAYSERA_ORDER_STATUS_PENDING',
            \Tools::getValue('PAYSERA_ORDER_STATUS_PENDING')
        );

        if ('PAYSERA_SUCCESS_STATUS_SETTING' === $this->module->l('PAYSERA_SUCCESS_STATUS_SETTING')) {
            $successMessage = 'Your Paysera settings were successfully updated.';
        } else {
            $successMessage = $this->module->l('PAYSERA_SUCCESS_STATUS_SETTING');
        }

        return new FormResult(true, $successMessage);
    }

    public function getFormFields()
    {
        $translations = $this->getTranslations();
        $statuses = $this->getOrderStatuses();

        return [[
            'form' => [
                'input' => [
                    $this->formHelper->getSelectForm(
                        'PAYSERA_ORDER_STATUS_NEW',
                        $translations['new_order'],
                        $statuses
                    ),
                    $this->formHelper->getSelectForm(
                        'PAYSERA_ORDER_STATUS_PAID',
                        $translations['paid_order'],
                        $statuses
                    ),
                    $this->formHelper->getSelectForm(
                        'PAYSERA_ORDER_STATUS_PENDING',
                        $translations['pending_order'],
                        $statuses
                    ),
                ],
                'submit' => [
                    'title' => $translations['save'],
                ],
            ],
        ]];
    }

    /**
     * @return array
     */
    public function getFieldsValue()
    {
        $configNewOrder = \Configuration::get('PAYSERA_ORDER_STATUS_NEW');
        $configPaidOrder = \Configuration::get('PAYSERA_ORDER_STATUS_PAID');
        $configPendingOrder = \Configuration::get('PAYSERA_ORDER_STATUS_PENDING');

        $statusSetting = [];
        $statusSetting['PAYSERA_ORDER_STATUS_NEW'] =
            \Tools::getValue('PAYSERA_ORDER_STATUS_NEW', $configNewOrder)
        ;
        $statusSetting['PAYSERA_ORDER_STATUS_PAID'] =
            \Tools::getValue('PAYSERA_ORDER_STATUS_PAID', $configPaidOrder)
        ;
        $statusSetting['PAYSERA_ORDER_STATUS_PENDING'] =
            \Tools::getValue('PAYSERA_ORDER_STATUS_PENDING', $configPendingOrder)
        ;

        return $statusSetting;
    }

    public function additionalHTML()
    {
        return '';
    }

    /**
     * @return array
     */
    private function getTranslations()
    {
        $translations = [];

        if ('PAYSERA_ORDER_STATUS_NEW' === $this->module->l('PAYSERA_ORDER_STATUS_NEW')) {
            $translations['new_order']['label'] = 'New order status';
        } else {
            $translations['new_order']['label'] = $this->module->l('PAYSERA_ORDER_STATUS_NEW');
        }
        if ('PAYSERA_ORDER_STATUS_PAID' === $this->module->l('PAYSERA_ORDER_STATUS_PAID')) {
            $translations['paid_order']['label'] = 'Paid order status';
        } else {
            $translations['paid_order']['label'] = $this->module->l('PAYSERA_ORDER_STATUS_PAID');
        }
        if ('PAYSERA_ORDER_STATUS_PENDING' === $this->module->l('PAYSERA_ORDER_STATUS_PENDING')) {
            $translations['pending_order']['label'] = 'Pending order status';
        } else {
            $translations['pending_order']['label'] = $this->module->l('PAYSERA_ORDER_STATUS_PENDING');
        }

        if ('PAYSERA_ORDER_STATUS_NEW_DESC' === $this->module->l('PAYSERA_ORDER_STATUS_NEW_DESC')) {
            $translations['new_order']['desc'] = 'Order status after finishing checkout';
        } else {
            $translations['new_order']['desc'] = $this->module->l('PAYSERA_ORDER_STATUS_NEW_DESC');
        }
        if ('PAYSERA_ORDER_STATUS_PAID_DESC' === $this->module->l('PAYSERA_ORDER_STATUS_PAID_DESC')) {
            $translations['paid_order']['desc'] = 'Status for order with confirmed payment';
        } else {
            $translations['paid_order']['desc'] = $this->module->l('PAYSERA_ORDER_STATUS_PAID_DESC');
        }
        if ('PAYSERA_ORDER_STATUS_PENDING_DESC' === $this->module->l('PAYSERA_ORDER_STATUS_PENDING_DESC')) {
            $translations['pending_order']['desc'] = 'Order status for pending payment';
        } else {
            $translations['pending_order']['desc'] = $this->module->l('PAYSERA_ORDER_STATUS_PENDING_DESC');
        }

        if ('BACKEND_STATUS_SAVE' === $this->module->l('BACKEND_STATUS_SAVE')) {
            $translations['save'] = 'Save status settings';
        } else {
            $translations['save'] = $this->module->l('BACKEND_STATUS_SAVE');
        }

        return $translations;
    }

    public function getFormId()
    {
        return $this->formId;
    }

    public function getFormTitle()
    {
        return $this->module->l('Status Settings');
    }

    public function getSubmitActionName()
    {
        return 'btnStatusSubmit';
    }

    /**
     * @return FormHelperInterface
     */
    public function getFormHelper()
    {
        return $this->formHelper;
    }

    /**
     * @return array
     */
    private function getOrderStatuses()
    {
        $states = [];
        $orderState = new \OrderState();
        foreach (\Language::getLanguages() as $language) {
            if ((int) $language['id_lang'] !== (int) \Configuration::get('PS_LANG_DEFAULT')) {
                continue;
            }

            $orderStateObj = $orderState->getOrderStates($language['id_lang']);

            foreach ($orderStateObj as $state) {
                $states[] = [
                    'id' => $state['id_order_state'],
                    'name' => $state['name'],
                ];
            }
        }

        return $states;
    }
}
