<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace Paysera\Scoped\Symfony\Component\Validator\Constraints;

use Paysera\Scoped\Symfony\Component\Intl\Currencies;
use Paysera\Scoped\Symfony\Component\Validator\Constraint;
use Paysera\Scoped\Symfony\Component\Validator\ConstraintValidator;
use Paysera\Scoped\Symfony\Component\Validator\Exception\LogicException;
use Paysera\Scoped\Symfony\Component\Validator\Exception\UnexpectedTypeException;
use Paysera\Scoped\Symfony\Component\Validator\Exception\UnexpectedValueException;
/**
 * Validates whether a value is a valid currency.
 *
 * @author Miha Vrhovnik <miha.vrhovnik@pagein.si>
 * @author Bernhard Schussek <bschussek@gmail.com>
 */
class CurrencyValidator extends ConstraintValidator
{
    /**
     * {@inheritdoc}
     */
    public function validate($value, Constraint $constraint)
    {
        if (!$constraint instanceof Currency) {
            throw new UnexpectedTypeException($constraint, Currency::class);
        }
        if (null === $value || '' === $value) {
            return;
        }
        if (!\is_scalar($value) && !(\is_object($value) && \method_exists($value, '__toString'))) {
            throw new UnexpectedValueException($value, 'string');
        }
        if (!\class_exists(Currencies::class)) {
            throw new LogicException('The Intl component is required to use the Currency constraint. Try running "composer require symfony/intl".');
        }
        $value = (string) $value;
        if (!Currencies::exists($value)) {
            $this->context->buildViolation($constraint->message)->setParameter('{{ value }}', $this->formatValue($value))->setCode(Currency::NO_SUCH_CURRENCY_ERROR)->addViolation();
        }
    }
}
