<?php
/**
 * 2018 Paysera
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/afl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * @author Paysera <plugins@paysera.com>
 * @copyright 2018 Paysera
 * @license http://opensource.org/licenses/afl-3.0.php Academic Free License (AFL 3.0)
 *  International Registered Trademark & Property of Paysera
 */
namespace Paysera\Helper;

if (!defined('_PS_VERSION_')) {
    exit;
}

use Paysera\Normalizer\TabNormalizer;

class TabHelper
{
    const PARENT_TAB_CLASS_NAME = 'AdminPayseraDeliveryParent';
    const PAYMENT_TAB_CLASS_NAME = 'AdminPayseraConfiguration';
    const DELIVERY_TAB_CLASS_NAME = 'AdminPayseraPaymentDeliveryTab';
    const ABOUT_TAB_CLASS_NAME = 'AdminPayseraPaymentAboutTab';
    const DELIVERY_MODULE_PAYMENT_TAB_CLASS_NAME = 'AdminPayseraDeliveryPayment';
    const DELIVERY_MODULE_DELIVERY_TAB_CLASS_NAME = 'AdminPayseraDelivery';
    const DELIVERY_MODULE_ABOUT_TAB_CLASS_NAME = 'AdminPayseraDeliveryAbout';
    const DELIVERY_MODULE_VERSION_TAB_CHANGE_VERSION = '1.5.1';
    /**
     * @var array<string, array<string, string|array>>
     */
    private $tabs;
    /**
     * @var array<string, array<string, string>>
     */
    private $deliveryModuleTabs;

    public function __construct()
    {
        $this->tabs = [
            [
                'name' => 'Paysera',
                'class_name' => self::PARENT_TAB_CLASS_NAME,
                'parent_id' => 'CONFIGURE',
                'delivery_module_tab' => '',
                'module' => 'paysera',
            ],
            [
                'name' => 'About',
                'class_name' => self::ABOUT_TAB_CLASS_NAME,
                'parent_id' => self::PARENT_TAB_CLASS_NAME,
                'module' => 'paysera',
                'languages' => [
                    'lt' => 'Apie',
                    'ru' => 'О нас',
                    'en' => 'About',
                ],
            ],
            [
                'name' => 'Payment',
                'class_name' => self::PAYMENT_TAB_CLASS_NAME,
                'parent_id' => self::PARENT_TAB_CLASS_NAME,
                'module' => 'paysera',
                'languages' => [
                    'lt' => 'Mokėjimai',
                    'ru' => 'Оплата',
                    'en' => 'Payment',
                    'lv' => 'Maksājums',
                    'pl' => 'Płatność',
                    'et' => 'Maksmine',
                    'es' => 'Pago',
                ],
            ],
            [
                'name' => 'Delivery',
                'class_name' => self::DELIVERY_TAB_CLASS_NAME,
                'parent_id' => self::PARENT_TAB_CLASS_NAME,
                'module' => 'paysera',
                'languages' => [
                    'lt' => 'Pristatymas',
                    'ru' => 'Доставка',
                    'en' => 'Delivery',
                ],
            ],
        ];

        $this->deliveryModuleTabs = [
            self::ABOUT_TAB_CLASS_NAME => self::DELIVERY_MODULE_ABOUT_TAB_CLASS_NAME,
            self::PAYMENT_TAB_CLASS_NAME => self::DELIVERY_MODULE_PAYMENT_TAB_CLASS_NAME,
            self::DELIVERY_TAB_CLASS_NAME => self::DELIVERY_MODULE_DELIVERY_TAB_CLASS_NAME,
        ];
    }

    public function installTabs()
    {
        $deliveryModule = null;

        if ($this->isDeliveryModuleEnabled()) {
            $deliveryModule = $this->getDeliveryModule();
        }

        foreach ($this->tabs as $tabSettings) {
            $existingTabId = \Tab::getIdFromClassName($tabSettings['class_name']);

            if ($existingTabId !== false) {
                continue;
            }

            if ($deliveryModule !== null) {
                if ($tabSettings['class_name'] === self::PAYMENT_TAB_CLASS_NAME) {
                    $tabSettings['parent_id'] = self::DELIVERY_MODULE_PAYMENT_TAB_CLASS_NAME;
                } elseif (in_array($tabSettings['class_name'], [self::DELIVERY_TAB_CLASS_NAME, self::ABOUT_TAB_CLASS_NAME], true)) {
                    continue;
                }
            }

            $tab = (new TabNormalizer())->mapToEntity($tabSettings);

            $tab->add();
        }
    }

    public function uninstallTabs()
    {
        foreach ($this->tabs as $tabSettings) {
            $tabId = \Tab::getIdFromClassName($tabSettings['class_name']);

            if ($tabId === false) {
                continue;
            }

            $tab = new \Tab($tabId);

            if (
                $tabSettings['class_name'] === self::PARENT_TAB_CLASS_NAME
                && $this->isDeliveryModuleEnabled()
            ) {
                $tab->module = \Paysera::PAYSERA_DELIVERY_MODULE_NAME;

                $tab->save();

                continue;
            }

            $tab->delete();
        }
    }

    public function syncTabs()
    {
        if ($this->isDeliveryModuleEnabled()) {
            $ownTabs = \Tab::getCollectionFromModule('paysera');
            $deliveryModule = $this->getDeliveryModule();

            foreach ($ownTabs as $tab) {
                if ($tab->class_name === self::PARENT_TAB_CLASS_NAME
                    && version_compare($deliveryModule->version, self::DELIVERY_MODULE_VERSION_TAB_CHANGE_VERSION, '>=')) {
                    continue;
                } elseif (array_key_exists($tab->class_name, $this->deliveryModuleTabs)) {
                    $tab = new \Tab(\Tab::getIdFromClassName($tab->class_name));
                    $tab->id_parent = \Tab::getIdFromClassName($this->deliveryModuleTabs[$tab->class_name]);

                    $tab->save();

                    continue;
                }

                $tab->delete();
            }
        } else {
            $this->uninstallTabs();
            $this->installTabs();
        }
    }

    /**
     * @return bool
     */
    private function isDeliveryModuleEnabled()
    {
        return \Module::isEnabled(\Paysera::PAYSERA_DELIVERY_MODULE_NAME);
    }

    /**
     * @return \Module
     */
    private function getDeliveryModule()
    {
        return \Module::getInstanceByName(\Paysera::PAYSERA_DELIVERY_MODULE_NAME);
    }
}
