<?php
/**
 * 2018 Paysera
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/afl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * @author Paysera <plugins@paysera.com>
 * @copyright 2018 Paysera
 * @license http://opensource.org/licenses/afl-3.0.php Academic Free License (AFL 3.0)
 *  International Registered Trademark & Property of Paysera
 */
namespace Paysera\Form;

if (!defined('_PS_VERSION_')) {
    exit;
}

use Paysera\Entity\FormResult;
use Paysera\Helper\FormHelperInterface;
use Paysera\Service\Provider\JSProvider;

class GeneralSettingsForm implements AdminFormInterface
{
    /**
     * @var FormHelperInterface
     */
    private $formHelper;
    /**
     * @var \Module
     */
    private $module;
    /**
     * @var string
     */
    private $formId;

    public function __construct(FormHelperInterface $formHelper, \Module $module)
    {
        $this->formHelper = $formHelper;
        $this->module = $module;
        $this->formId = 'GeneralSettingsForm';
    }

    /**
     * @return FormResult
     */
    public function validate()
    {
        $warning = '';
        $projectId = trim((string) \Tools::getValue('PAYSERA_GENERAL_PROJECT_ID'));
        $projectSign = trim((string) \Tools::getValue('PAYSERA_GENERAL_SIGN_PASS'));

        if ($projectId === '' && $projectSign === '') {
            $warning = $this->module->l('Project ID and Sign are required, please fill out these fields.');
        } elseif ($projectId === '') {
            $warning = $this->module->l('Project ID is required, please fill out this field.');
        } elseif ($projectSign === '') {
            $warning = $this->module->l('Project Sign is required, please fill out this field.');
        } elseif (!is_numeric($projectId)) {
            $warning = $this->module->l('Project ID must consist of numbers only. Please provide a valid Project ID.');
        }

        return new FormResult($warning === '', $warning);
    }

    /**
     * @return FormResult
     */
    public function update()
    {
        \Configuration::updateValue(
            'PAYSERA_GENERAL_PROJECT_ID',
            \Tools::getValue('PAYSERA_GENERAL_PROJECT_ID')
        );
        \Configuration::updateValue(
            'PAYSERA_GENERAL_SIGN_PASS',
            \Tools::getValue('PAYSERA_GENERAL_SIGN_PASS')
        );
        \Configuration::updateValue(
            'PAYSERA_GENERAL_TEST_MODE',
            \Tools::getValue('PAYSERA_GENERAL_TEST_MODE')
        );

        if ('PAYSERA_SUCCESS_GENERAL_SETTING' === $this->module->l('PAYSERA_SUCCESS_GENERAL_SETTING')) {
            $successMessage = 'Your Paysera settings were successfully updated.';
        } else {
            $successMessage = $this->module->l('PAYSERA_SUCCESS_GENERAL_SETTING');
        }

        return new FormResult(true, $successMessage);
    }

    public function getFormFields()
    {
        $translations = $this->getTranslations();

        return [[
            'form' => [
                'input' => [
                    $this->formHelper->getTextForm(
                        'PAYSERA_GENERAL_PROJECT_ID',
                        $translations['project_id'],
                        true
                    ),
                    $this->formHelper->getTextForm(
                        'PAYSERA_GENERAL_SIGN_PASS',
                        $translations['sign_pass'],
                        true
                    ),
                    $this->formHelper->getBoolForm(
                        'PAYSERA_GENERAL_TEST_MODE',
                        $translations['test']
                    ),
                ],
                'submit' => [
                    'title' => $translations['save'],
                ],
            ],
        ]];
    }

    /**
     * @return array
     */
    public function getFieldsValue()
    {
        $configProjectID = \Configuration::get('PAYSERA_GENERAL_PROJECT_ID');
        $configSignPass = \Configuration::get('PAYSERA_GENERAL_SIGN_PASS');
        $configTest = \Configuration::get('PAYSERA_GENERAL_TEST_MODE');

        $generalSetting = [];
        $generalSetting['PAYSERA_GENERAL_PROJECT_ID'] =
            \Tools::getValue('PAYSERA_GENERAL_PROJECT_ID', $configProjectID)
        ;
        $generalSetting['PAYSERA_GENERAL_SIGN_PASS'] =
            \Tools::getValue('PAYSERA_GENERAL_SIGN_PASS', $configSignPass)
        ;
        $generalSetting['PAYSERA_GENERAL_TEST_MODE'] =
            \Tools::getValue('PAYSERA_GENERAL_TEST_MODE', $configTest)
        ;

        return $generalSetting;
    }

    public function additionalHTML()
    {
        return (new JSProvider())->getNumberOnlyInputEventListener('#PAYSERA_GENERAL_PROJECT_ID');
    }

    /**
     * @return array
     */
    private function getTranslations()
    {
        $translations = [];

        if ('PAYSERA_GENERAL_PROJECT_ID' === $this->module->l('PAYSERA_GENERAL_PROJECT_ID')) {
            $translations['project_id']['label'] = 'Project ID';
        } else {
            $translations['project_id']['label'] = $this->module->l('PAYSERA_GENERAL_PROJECT_ID');
        }
        if ('PAYSERA_GENERAL_SIGN_PASS' === $this->module->l('PAYSERA_GENERAL_SIGN_PASS')) {
            $translations['sign_pass']['label'] = 'Sign';
        } else {
            $translations['sign_pass']['label'] = $this->module->l('PAYSERA_GENERAL_SIGN_PASS');
        }
        if ('PAYSERA_GENERAL_TEST_MODE' === $this->module->l('PAYSERA_GENERAL_TEST_MODE')) {
            $translations['test']['label'] = 'Test';
        } else {
            $translations['test']['label'] = $this->module->l('PAYSERA_GENERAL_TEST_MODE');
        }

        if ('PAYSERA_GENERAL_PROJECT_ID_DESC' === $this->module->l('PAYSERA_GENERAL_PROJECT_ID_DESC')) {
            $translations['project_id']['desc'] = 'Paysera project id';
        } else {
            $translations['project_id']['desc'] = $this->module->l('PAYSERA_GENERAL_PROJECT_ID_DESC');
        }
        if ('PAYSERA_GENERAL_SIGN_PASS_DESC' === $this->module->l('PAYSERA_GENERAL_SIGN_PASS_DESC')) {
            $translations['sign_pass']['desc'] = 'Paysera project password';
        } else {
            $translations['sign_pass']['desc'] = $this->module->l('PAYSERA_GENERAL_SIGN_PASS_DESC');
        }
        if ('PAYSERA_GENERAL_TEST_MODE_DESC' === $this->module->l('PAYSERA_GENERAL_TEST_MODE_DESC')) {
            $translations['test']['desc'] = 'Enable this to accept test payments';
        } else {
            $translations['test']['desc'] = $this->module->l('PAYSERA_GENERAL_TEST_MODE_DESC');
        }

        if ('BACKEND_GENERAL_SAVE' === $this->module->l('BACKEND_GENERAL_SAVE')) {
            $translations['save'] = 'Save main settings';
        } else {
            $translations['save'] = $this->module->l('BACKEND_GENERAL_SAVE');
        }

        return $translations;
    }

    public function getFormId()
    {
        return $this->formId;
    }

    public function getFormTitle()
    {
        return $this->module->l('Main Settings');
    }

    public function getSubmitActionName()
    {
        return 'btnGeneralSubmit';
    }

    /**
     * @return FormHelperInterface
     */
    public function getFormHelper()
    {
        return $this->formHelper;
    }
}
