<?php
/**
 * 2018 Paysera
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/afl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * @author Paysera <plugins@paysera.com>
 * @copyright 2018 Paysera
 * @license http://opensource.org/licenses/afl-3.0.php Academic Free License (AFL 3.0)
 * International Registered Trademark & Property of Paysera
 */
declare(strict_types=1);

namespace Paysera\Delivery\HookHandler\OrderHandler;

use Paysera\Delivery\HookHandler\OrderHandlerInterface;
use Paysera\Delivery\HookHandler\HookHandlerParameters;
use Paysera\Delivery\Service\Logger;
use Paysera\Delivery\Service\DeliveryApiClient;
use Paysera\Delivery\Model\ShipmentRequestModel;
use Paysera\Delivery\Manager\ShipmentRequestManager;
use Paysera\Delivery\Entity\ShipmentRequest;
use Order;
use Validate;

class PayseraPaymentConfirmedHandler implements OrderHandlerInterface
{
    /**
     * @var
     */
    private $logger;
    /**
     * @var DeliveryApiClient
     */
    private $deliveryApiClient;
    /**
     * @var ShipmentRequestModel
     */
    private $shipmentRequestModel;
    /**
     * @var ShipmentRequestManager
     */
    private $shipmentRequestManager;

    public function __construct(
        Logger $logger,
        DeliveryApiClient $deliveryApiClient,
        ShipmentRequestModel $shipmentRequestModel,
        ShipmentRequestManager $shipmentRequestManager
    ) {
        $this->logger = $logger;
        $this->deliveryApiClient = $deliveryApiClient;
        $this->shipmentRequestModel = $shipmentRequestModel;
        $this->shipmentRequestManager = $shipmentRequestManager;
    }

    /**
     * Handles the Paysera payment confirmed hook.
     *
     * @param HookHandlerParameters $hookHandlerParameters
     * @return void
     */
    public function handle(HookHandlerParameters $hookHandlerParameters): void
    {
        $params = $hookHandlerParameters->getParams();

        /** @var Order $order */
        $order = $params['order'];

        if (!Validate::isLoadedObject($order) || $order->id === null) {
            $this->logger->error('Order not found. Order ID: ' . ($order->id ?? 'null'));

            return;
        }

        $shipmentRequest = $this->shipmentRequestModel->find($order->id);

        if ($shipmentRequest === null) {
            $this->logger->info('Shipment request not found for Order ID: ' . $order->id);

            return;
        }

        if ($shipmentRequest->getStatus() !== ShipmentRequest::STATUS_NEW) {
            $this->logger->info('Delivery already ' . $shipmentRequest->getStatus() . ' for Order ID: ' . $order->id);

            return;
        }

        $this->deliveryApiClient->setOrderStatusPrepaid($shipmentRequest);
        $this->shipmentRequestManager->markAsPaid($shipmentRequest);
    }
}
