<?php

namespace Opencart\Catalog\Controller\Extension\PayseraDelivery\Shipping;

use Opencart\System\Engine\Controller;
use Paysera\Delivery\DeliveryManager;
use Paysera\Delivery\Entity\ShipmentRequest;
use Paysera\Delivery\Entity\ShipmentRequestGateway;
use Paysera\Delivery\Exception\ProductNotFoundException;
use Paysera\Delivery\Exception\ShipmentRequestGatewayNotFoundException;
use Paysera\Delivery\Manager\ShipmentRequestManager;
use Paysera\Delivery\Model\OrderModel;
use Paysera\Delivery\Model\ProductModel;
use Paysera\Delivery\Model\ShipmentRequestModel;
use Paysera\Delivery\Normalizer\OrderNormalizer;
use Paysera\Delivery\Normalizer\ProductNormalizer;
use Paysera\Delivery\Normalizer\ShipmentRequestNormalizer;
use Paysera\Delivery\PayseraDeliveryLibraryHelper;
use Paysera\Delivery\PayseraDeliveryResolver;
use Paysera\Delivery\PayseraDeliverySettings;
use Paysera\Delivery\Service\DeliveryOrderHistoryGenerator;
use Paysera\Delivery\Service\Helper\ShipmentRequestHelper;
use Paysera\Delivery\Service\Logger;
use Paysera\Delivery\Service\RoutePathResolver;

if (is_file(DIR_EXTENSION . 'paysera_delivery/system/library/paysera/delivery/vendor/autoload.php')) {
    require_once DIR_EXTENSION . 'paysera_delivery/system/library/paysera/delivery/vendor/autoload.php';
}

class Paysera extends Controller
{
    const PAYSERA_DELIVERY_TERMINAL = 'extension/paysera_delivery/shipping/terminal';
    const CONTENT_TYPE_APPLICATION_JSON = 'Content-Type: application/json';
    const PARAM_DELIVERY_GATEWAY_CODE = 'delivery_gateway_code';
    const ORDER_STATUS_PENDING_ID = 1;

    private $delivery;
    private $deliveryResolver;
    private $deliveryManager;

    private ShipmentRequestModel $shipmentRequestModel;
    private ShipmentRequestManager $shipmentRequestManager;
    private ShipmentRequestHelper $shipmentRequestHelper;
    private OrderModel $orderModel;
    private Logger $logger;

    public function __construct($registry)
    {
        parent::__construct($registry);

        $this->load->language(PayseraDeliverySettings::LANGUAGE_SETTINGS);

        $this->delivery = new PayseraDeliveryLibraryHelper($this->registry);
        $this->deliveryResolver = new PayseraDeliveryResolver();
        $this->deliveryManager = (new DeliveryManager($registry))->load();

        $this->load->model('checkout/order');
        $this->load->model('catalog/product');
        $this->load->model('extension/paysera_delivery/shipping/paysera');

        $this->shipmentRequestModel = new ShipmentRequestModel(
            $this->model_extension_paysera_delivery_shipping_paysera,
            new ShipmentRequestNormalizer()
        );
        $this->shipmentRequestManager = new ShipmentRequestManager($this->shipmentRequestModel);
        $this->shipmentRequestHelper = new ShipmentRequestHelper(
            $this->shipmentRequestModel
        );
        $this->productModel = new ProductModel(
            $this->model_catalog_product,
            new ProductNormalizer()
        );
        $this->orderModel = new OrderModel(
            $this->model_checkout_order,
            $this->productModel,
            new OrderNormalizer(),
        );
        $this->logger = new Logger($this->log);
    }

    public function shippingMethods(&$route, &$data, &$output)
    {
        $this->load->language('extension/paysera_delivery/shipping/paysera');

        $data['choose_terminal'] = $this->language->get('label_terminal');
        $data['entry_country_default'] = $this->language->get('entry_country');
        $data['route_action_separator'] = (new RoutePathResolver(VERSION))->getRouteActionSeparator();

        $data['selected_terminal_data'] = $this->getSelectedTerminalData();

        $output = $this->load->view('extension/paysera_delivery/shipping/paysera', $data) . $output;
    }

    public function getCountryByGatewayCode()
    {
        $this->load->model('localisation/country');
        $this->load->language('extension/paysera_delivery/shipping/paysera');
        $this->load->model('extension/paysera_delivery/shipping/paysera');
        $this->load->model('localisation/zone');

        $deliveryGatewayCode = $this->request->get[self::PARAM_DELIVERY_GATEWAY_CODE];
        $deliveryGatewaySettings = $this->deliveryManager->getGatewaySettings(
            $this->getDeliveryGatewayCode($deliveryGatewayCode),
            $this->getDeliveryGatewayIndex($deliveryGatewayCode)
        );

        $data['countries'] = [];
        if (!empty($deliveryGatewaySettings->getGeoZone())) {
            $geoZoneInfo = $this->model_extension_paysera_delivery_shipping_paysera->getGeoZone($deliveryGatewaySettings->getGeoZone());
            foreach ($geoZoneInfo as $geoZone) {
                $data['countries'][] = $this->model_localisation_country->getCountry($geoZone['country_id']);
            }
            $data['countries'] = array_unique($data['countries'], SORT_REGULAR);
        } else {
            foreach ($this->delivery->getCountries($this->getDeliveryGatewayCode($deliveryGatewayCode)) as $country) {
                $data['countries'][] = $this->model_extension_paysera_delivery_shipping_paysera->getCountryByCode($country);
            }
        }

        $deliveryCountryIsoCode2 = $this->getCountryByIsoCode2()
            ?? $this->session->data['shipping_address']['iso_code_2'];
        foreach ($data['countries'] as $country) {
            if (in_array($deliveryCountryIsoCode2, $country, true)) {
                $data['delivery_country_id'] = $country['country_id'];
                $deliveryGateway = $this->delivery->getDeliveryGatewayByCode($this->getDeliveryGatewayCode($deliveryGatewayCode));
                $cities = $this->delivery->getCities($deliveryGateway, $country['iso_code_2']);

                $city = $this->session->data['shipping_address']['city'] ?? '';
                $filteredCity = array_filter($cities, function ($c) use ($city) {
                    $c = preg_replace("/[\x{0300}-\x{036f}]/u", '', \Normalizer::normalize($c, \Normalizer::NFD));
                    return strtolower($c) === strtolower($city);
                });
                $data['delivery_city'] = (count($filteredCity) > 0 ? $city : null);
            }
        }

        $data['choose_terminal'] = $this->language->get('label_terminal');
        $data['entry_country_default'] = $this->language->get('entry_country');

        $output = $this->load->view($this::PAYSERA_DELIVERY_TERMINAL, $data);

        $this->outputResponse($output);
    }

    public function validate()
    {
        $this->load->language(PayseraDeliverySettings::LANGUAGE_SETTINGS);

        $this->load->model('setting/setting');

        $deliveryGatewayCode = $this->request->get[self::PARAM_DELIVERY_GATEWAY_CODE];

        $deliveryGateway = $this->deliveryManager->getGatewaySettings(
            $this->getDeliveryGatewayCode($deliveryGatewayCode),
            $this->getDeliveryGatewayIndex($deliveryGatewayCode)
        );
        $json = [];

        if ($deliveryGateway === null) {
            $json['error'] = $this->language->get('error_shipping_method_not_exist');
        } else {
            if (in_array(
                $deliveryGateway->getReceiverType(),
                [
                    PayseraDeliverySettings::TYPE_PARCEL_MACHINE,
                    PayseraDeliverySettings::TYPE_COURIER
                ],
                true
            )) {
                $defaultUnit = $this->weight->getUnit($this->config->get('config_weight_class_id'));

                $totalWeight = 0.0;
                foreach ($this->cart->getProducts() as $product) {
                    $convertedProductWeight = (float) $this->weight->convert(
                        $product['weight'],
                        $product['weight_class_id'],
                        $defaultUnit
                    );
                    $totalWeight += $convertedProductWeight;
                }

                if ($deliveryGateway->getReceiverType() === PayseraDeliverySettings::TYPE_PARCEL_MACHINE) {
                    $json['has_delivery_terminal'] = 1;
                }

                if ($totalWeight > (int) $deliveryGateway->getMaximumWeight()) {
                    $json['error'] = sprintf(
                        $this->language->get('error_max_weight'),
                        $totalWeight,
                        $deliveryGateway->getMaximumWeight(),
                        $defaultUnit
                    );
                }

                if ($totalWeight < (int) $deliveryGateway->getMinimumWeight()) {
                    $json['error'] = sprintf(
                        $this->language->get('error_min_weight'),
                        $totalWeight,
                        $deliveryGateway->getMinimumWeight(),
                        $defaultUnit
                    );
                }
            } else {
                $json['has_delivery_terminal'] = 0;
            }
        }

        $this->outputResponse($json);
    }

    public function cities()
    {
        $this->load->model('localisation/country');

        $deliveryGatewayCode = $this->request->get[self::PARAM_DELIVERY_GATEWAY_CODE];
        $countryId = $this->request->get['country_id'] ?? false;
        $countryInfo = $this->model_localisation_country->getCountry($countryId);

        $deliveryGateway = $this->delivery->getDeliveryGatewayByCode($this->getDeliveryGatewayCode($deliveryGatewayCode));

        $cities = $this->delivery->getCities($deliveryGateway, $countryInfo['iso_code_2']);

        array_unshift($cities, ['default' => $this->language->get('entry_city')]);

        if (isset($this->session->data['delivery_gateway_terminal'])) {
            unset($this->session->data['delivery_gateway_terminal']);
            unset($this->session->data['delivery_gateway_country_id']);
            unset($this->session->data['delivery_gateway_city']);
            unset($this->session->data['delivery_gateway_zone_code']);
        }

        $this->outputResponse($cities);
    }

    public function terminals()
    {
        $this->load->model('localisation/country');

        $deliveryGatewayCode = $this->request->get[self::PARAM_DELIVERY_GATEWAY_CODE];
        $countryId = $this->request->get['country_id'] ?? false;
        $city = $this->request->get['city'] ?? false;
        $countryInfo = $this->model_localisation_country->getCountry($countryId);

        $deliveryGateway = $this->delivery->getDeliveryGatewayByCode($this->getDeliveryGatewayCode($deliveryGatewayCode));

        $terminals = $this->delivery->getTerminalByLocation(
            $deliveryGateway,
            $countryInfo['iso_code_2'],
            $city
        );

        if (isset($this->session->data['delivery_gateway_terminal'])) {
            unset($this->session->data['delivery_gateway_terminal']);
            unset($this->session->data['delivery_gateway_country_id']);
            unset($this->session->data['delivery_gateway_city']);
            unset($this->session->data['delivery_gateway_zone_code']);
        }

        $this->outputResponse(array_merge(['default' => $this->language->get('entry_terminal')], $terminals));
    }

    public function saveDeliveryGateway()
    {
        $this->load->model('localisation/country');
        $this->load->model('localisation/zone');

        $countryInfo = $this->model_localisation_country->getCountry($this->request->post['country_id']);
        $zonesInfo = $this->model_localisation_zone->getZonesByCountryId($this->request->post['country_id']);
        $zoneCode = null;
        foreach ($zonesInfo as $zone) {
            if ($zone['name'] === $this->request->post['city']) {
                $zoneCode = $zone['code'];
            }
        }
        $this->session->data['delivery_gateway_terminal'] = $this->request->post['terminal'];
        $this->session->data['delivery_gateway_country_iso_code_2'] = $countryInfo['iso_code_2'];
        $this->session->data['delivery_gateway_city'] = $this->request->post['city'];
        $this->session->data['delivery_gateway_zone_code'] = $zoneCode;
    }

    public function createDeliveryOrder(&$route, &$args)
    {
        $settings = $this->deliveryManager->getSettings();

        if ($settings->isTestModeEnabled() === true) {
            $this->log->write('Test mode is enabled.');

            return;
        }

        $orderId = $this->session->data['order_id'] ?? null;
        $orderStatus = $this->session->data['order_status_id'] ?? self::ORDER_STATUS_PENDING_ID;

        if ($orderId === null) {
            return;
        }

        [$terminal, $countryIsoCode2, $city, $zoneCode, $shippingMethod] = $this->getShippingParametersAndClearSession();
        if (empty($shippingMethod)) {
            return;
        }

        $gateway = null;
        if ($terminal !== null || $countryIsoCode2 !== null || $city !== null || $zoneCode !== null) {
            $gateway = new ShipmentRequestGateway($terminal, $countryIsoCode2, $city, $zoneCode);
        }

        if ($shippingMethod !== null) {
            $this->shipmentRequestHelper->saveFromProperties(
                $orderId,
                $shippingMethod,
                $gateway
            );
        }

        $this->load->language(PayseraDeliverySettings::LANGUAGE_SETTINGS);

        $order = $this->orderModel->getOrder($orderId);
        if ($order === null) {
            $this->logger->log('Order not found');

            return;
        }

        $shipmentRequest = $this->shipmentRequestModel->get($orderId);
        if ($shipmentRequest === null) {
            $this->logger->log('Shipment request not found');

            return;
        } else if ($shipmentRequest->getStatus() === ShipmentRequest::STATUS_COMPLETED) {
            return;
        }

        $deliveryGatewaySettings = $this->deliveryManager->getGatewaySettings(
            $this->deliveryResolver->resolveDeliveryGatewayCode($shipmentRequest->getShippingMethod()),
            $this->deliveryResolver->resolveDeliveryGatewayIndex($shipmentRequest->getShippingMethod())
        );

        $createdDeliveryOrder = null;
        try {
            $createdDeliveryOrder = $this->delivery->createDeliveryOrder(
                $deliveryGatewaySettings,
                $this->orderModel->getProductsForOrder($orderId),
                $order,
                $shipmentRequest
            );
        } catch (ShipmentRequestGatewayNotFoundException $exception) {
            $this->logger->log('Shipment request gateway not found');
        } catch (ProductNotFoundException $exception) {
            $this->logger->log('Product not found');
        }

        if ($createdDeliveryOrder === null) {
            return;
        }

        $this->shipmentRequestManager->markAsCompleted($shipmentRequest);
        $this->logger->log(sprintf($this->language->get('order_created_note'), $createdDeliveryOrder->getNumber()));

        $deliveryOrderHistoryGenerator = new DeliveryOrderHistoryGenerator(
            $this->language,
            $this->model_checkout_order
        );

        $deliveryOrderHistoryGenerator->generateDefaultOrderHistory(
            $orderId,
            $createdDeliveryOrder->getNumber(),
            $orderStatus
        );

        if ($deliveryGatewaySettings->getReceiverType() === PayseraDeliverySettings::TYPE_PARCEL_MACHINE) {
            $deliveryOrderHistoryGenerator->generateTerminalDeliveryOrderHistory(
                $orderId,
                $orderStatus,
                $createdDeliveryOrder->getNumber(),
                $createdDeliveryOrder->getShipmentGateway()->getDescription(),
                $createdDeliveryOrder->getShipmentMethod()->getDescription(),
                $createdDeliveryOrder->getReceiver()->getParcelMachine()->getAddress()->getStreet() ?? '',
                $createdDeliveryOrder->getReceiver()->getParcelMachine()->getLocationName()
            );
        }
    }

    public function shippingMethodSave(): void
    {
        if (isset($this->request->post['shipping_method'])) {
            $shipping = explode('.', $this->request->post['shipping_method']);
            if (
                !isset($shipping[0])
                || !isset($shipping[1])
                || !isset($this->session->data['shipping_methods'][$shipping[0]]['quote'][$shipping[1]])
            ) {
                return;
            }

            $pattern = '/<img\s[^>]*>/i';
            $nameWithoutLogo = preg_replace(
                $pattern,
                '',
                $this->session->data['shipping_methods'][$shipping[0]]['quote'][$shipping[1]]['name']
            );

            $this->session->data['shipping_methods'][$shipping[0]]['quote'][$shipping[1]]['name'] = $nameWithoutLogo;
        }

        unset($this->session->data['delivery_gateway_country_iso_code_2']);
        unset($this->session->data['delivery_gateway_city']);
        unset($this->session->data['delivery_gateway_terminal']);
        unset($this->session->data['delivery_gateway_zone_code']);
    }

    protected function getShippingParametersAndClearSession(): array
    {
        $terminal = $this->session->data['delivery_gateway_terminal'] ?? null;
        $countryIsoCode2 = $this->session->data['delivery_gateway_country_iso_code_2'] ?? null;
        $city = $this->session->data['delivery_gateway_city'] ?? null;
        $zoneCode = $this->session->data['delivery_gateway_zone_code'] ?? null;

        $shippingMethod = is_string($this->session->data['shipping_method'])
            ? $this->session->data['shipping_method'] // OpenCart < 4.0.2.0
            : $this->session->data['shipping_method']['code'] ?? null; // OpenCart >= 4.0.2.0

        if (empty($shippingMethod) || !str_starts_with($shippingMethod, 'paysera.')) {
            $shippingMethod = null;
        } else {
            $shippingMethod = str_replace('paysera.', '', $shippingMethod ?? '');
        }

        unset(
            $this->session->data['delivery_gateway_terminal'],
            $this->session->data['delivery_gateway_country_iso_code_2'],
            $this->session->data['delivery_gateway_city'],
            $this->session->data['delivery_gateway_zone_code'],
            $this->session->data['shipping_method'],
            $this->session->data['order_status_id'],
        );

        return [
            $terminal,
            $countryIsoCode2,
            $city,
            $zoneCode,
            $shippingMethod
        ];
    }

    private function outputResponse($response)
    {
        $this->response->addHeader(self::CONTENT_TYPE_APPLICATION_JSON);
        $this->response->setOutput(json_encode($response));
    }

    /**
     * @param string $param
     * @return string
     */
    private function getDeliveryGatewayCode($param)
    {
        $deliveryGatewayArray = explode('_', $param);

        return current($deliveryGatewayArray);
    }

    /**
     * @param string $param
     * @return string
     */
    private function getDeliveryGatewayIndex($param)
    {
        $deliveryGatewayArray = explode('_', $param);

        return end($deliveryGatewayArray);
    }

    protected function getSelectedTerminalData(): array
    {
        if ($this->session->data['delivery_gateway_country_iso_code_2'] ?? null) {
            return [
                'country' => $this->getCountryByIsoCode2(),
                'city' => $this->session->data['delivery_gateway_city'] ?? null,
                'terminal' => $this->session->data['delivery_gateway_terminal'] ?? null,
            ];
        }

        return [
            'country' => null,
            'city' => null,
            'terminal' => null,
        ];
    }

    protected function getCountryByIsoCode2(): ?string
    {
        $this->load->model('localisation/country');
        $country = $this->model_localisation_country
            ->getCountryByIsoCode2($this->session->data['delivery_gateway_country_iso_code_2'] ?? 0);

        return $country['country_id'] ?? $country[0]['country_id'] ?? null;
    }
}
