<?php

namespace Opencart\Admin\Controller\Extension\PayseraDelivery\Shipping;

use Opencart\System\Engine\Controller;
use Paysera\Delivery\DeliveryManager;
use Paysera\Delivery\Entity\DeliveryGatewaySettings;
use Paysera\Delivery\Normalizer\DeliveryGatewaySettingsNormalizer;
use Paysera\Delivery\Normalizer\DeliverySettingsNormalizer;
use Paysera\Delivery\Normalizer\ShipmentGatewayNormalizer;
use Paysera\Delivery\Normalizer\ShipmentMethodNormalizer;
use Paysera\Delivery\PayseraDeliveryLibraryHelper;
use Paysera\Delivery\PayseraDeliveryResolver;
use Paysera\Delivery\PayseraDeliverySettings;
use Paysera\Delivery\Repository\MessageRepository;
use Paysera\Delivery\Service\RoutePathResolver;
use Paysera\Delivery\ShipmentGatewayManager;
use Paysera\Delivery\Validator\DeliveryGatewayValidator;
use Paysera\DeliveryApi\MerchantClient\Entity\ShipmentGateway;

if (is_file(DIR_EXTENSION . 'paysera_delivery/system/library/paysera/delivery/vendor/autoload.php')) {
    require_once DIR_EXTENSION . 'paysera_delivery/system/library/paysera/delivery/vendor/autoload.php';
}

class Paysera extends Controller
{
    const PAYSERA_EVENT_MENU_NAME = 'paysera_delivery_menu';
    const PAYSERA_EVENT_MENU = 'admin/view/common/column_left/before';
    const PAYSERA_EXTENSION_DELIVERY = 'extension/paysera_delivery/shipping/paysera';
    const PAYSERA_EXTENSION_DELIVERY_VIEW = 'extension/paysera_delivery/shipping/delivery';
    const PAYSERA_EXTENSION_ABOUT_VIEW = 'extension/paysera_delivery/shipping/about';
    const PAYSERA_EXTENSION_NO_PAYMENT_PLUGIN_VIEW = 'extension/paysera_delivery/shipping/no_payment_plugin';
    const PAYSERA_EXTENSION_DELIVERY_GATEWAY_VIEW = 'extension/paysera_delivery/shipping/delivery_gateway';
    const PAYSERA_EXTENSION_SHIPPING_METHODS_VEW = 'extension/paysera_delivery/shipping/shipping_methods';
    const PAYSERA_EXTENSION_DELIVERY_TAB_MAIN_SETTINGS = 'extension/paysera_delivery/shipping/main_settings_form';
    const PAYSERA_EXTENSION_DELIVERY_TAB_GATEWAY_LIST = 'extension/paysera_delivery/shipping/delivery_gateway_list';
    const PAYSERA_EXTENSION_PAYMENT_LINK = 'https://www.opencart.com/index.php?route=marketplace/extension/info&extension_id=15090&filter_category_id=3';
    const PAYSERA_EMPTY_VALUE = '';
    const REQUEST_METHOD_TYPE = 'POST';
    const PAYSERA_SUCCESS = 'text_success';
    const PAYSERA_ERROR = 'text_error';
    const PAYSERA_ERROR_PREFIX = 'error_';
    const PAYSERA_SESSION_PREFIX = 'session_';
    const PAYSERA_DELIVERY_GATEWAY_LIST = 'delivery_gateway_list';

    const PAYSERA_DELIVERY_SHIPPING_METHOD_NAME = 'paysera_delivery_shipping_method';
    const PAYSERA_DELIVERY_EVENT_SHIPPING_METHOD = 'catalog/view/checkout/shipping_method/after';
    const PAYSERA_DELIVERY_CREATE_ORDER_NAME = 'paysera_delivery_create_order';
    const PAYSERA_DELIVERY_EVENT_ADD_ORDER  = 'catalog/controller/checkout/success/before';
    const PAYSERA_SHIPPING_METHOD_SAVE_NAME = 'paysera_shipping_method_save_before';
    const PAYSERA_SHIPPING_METHOD_SAVE  = 'catalog/controller/checkout/shipping_method.save/before';

    const PAYSERA_EXTENSION_MENU_ROUTE_ACTION = 'payseraDeliveryMenuHandler';
    const PAYSERA_EXTENSION_ABOUT_ROUTE_ACTION = 'about';
    const PAYSERA_EXTENSION_NO_PAYMENT_PLUGIN_ROUTE_ACTION = 'no_payment_plugin';
    const PAYSERA_EXTENSION_DELIVERY_GATEWAY_ROUTE_ACTION = 'delivery_gateway';
    const PAYSERA_EXTENSION_SHIPPING_METHODS_ROUTE_ACTION = 'shippingMethods';
    const PAYSERA_EXTENSION_DELIVERY_GATEWAY_ADD_REMOVE_ROUTE_ACTION = 'addRemoveDeliveryGateway';
    const PAYSERA_EXTENSION_CREATE_DELIVERY_ROUTE_ACTION = 'createDeliveryOrder';
    const PAYSERA_SHIPPING_METHOD_SAVE_ROUTE_ACTION = 'shippingMethodSave';

    const PAYSERA_PARAM_ACTION = 'action';
    const PAYSERA_PARAM_CODE = 'code';
    const PAYSERA_PARAM_KEY = 'key';
    const PAYSERA_PARAM_USER_TOKEN = 'user_token';

    const PAYSERA_ACTION_ADD = 'add';
    const PAYSERA_ACTION_REMOVE = 'remove';
    const OPENCART_ORDER_STATUS_PROCESSING = 2;

    private $error = [];
    private $errorFieldName = [
        'warning',
        'project_id',
        'project_password',
    ];
    private $sessionFieldName = [
        'success',
        'error_warning',
    ];
    private $breadcrumbFields = [
        'text_home' => 'common/dashboard',
        'text_extension' => 'marketplace/extension',
        'heading_title' => 'extension/paysera_delivery/shipping/paysera',
    ];
    private $payseraFieldsName = [
        PayseraDeliverySettings::STATUS,
        PayseraDeliverySettings::TEST_MODE,
        PayseraDeliverySettings::DELIVERY_GATEWAY,
        PayseraDeliverySettings::PROJECT_ID,
        PayseraDeliverySettings::PROJECT_PASSWORD,
        PayseraDeliverySettings::FREE_DELIVERY_THRESHOLD,
    ];
    private $delivery;
    private $deliveryHelper;
    private $shipmentGatewayNormalizer;
    private $deliverySettingsNormalizer;
    private $deliveryGatewaySettingsNormalizer;
    private $deliveryManager;
    private $shipmentGatewayManager;
    private RoutePathResolver $routePathResolver;

    public function __construct($registry)
    {
        parent::__construct($registry);

        $this->load->model('setting/setting');
        $this->load->model('localisation/currency');
        $this->load->language(PayseraDeliverySettings::LANGUAGE_SETTINGS);

        $this->delivery = new PayseraDeliveryLibraryHelper($registry);
        $this->shipmentMethodNormalizer = new ShipmentMethodNormalizer();
        $this->shipmentGatewayNormalizer = new ShipmentGatewayNormalizer();
        $this->deliverySettingsNormalizer = new DeliverySettingsNormalizer();
        $this->deliveryGatewaySettingsNormalizer = new DeliveryGatewaySettingsNormalizer();
        $this->deliveryManager = (new DeliveryManager($registry))->load();
        $this->shipmentGatewayManager = (new ShipmentGatewayManager($registry, $this->delivery));
        $this->deliveryHelper = new PayseraDeliveryResolver();
        $this->routePathResolver = new RoutePathResolver(VERSION);
    }

    public function index()
    {
        $this->deliveryManager->syncDeliveryGateways($this->delivery->getDeliveryGateways());

        $this->document->setTitle($this->getText('heading_title'));

        $settings = $this->deliveryManager->getSettings();

        if ($this->request->server['REQUEST_METHOD'] === self::REQUEST_METHOD_TYPE && $this->validate()) {
            $postSettings = $this->request->post;
            $postSettings[PayseraDeliverySettings::PROJECT_HASH] = $this->delivery->getProjectHash(
                $postSettings[PayseraDeliverySettings::PROJECT_ID],
                $postSettings[PayseraDeliverySettings::PROJECT_PASSWORD]
            );

            if ($postSettings[PayseraDeliverySettings::PROJECT_HASH] !== null) {
                $this->deliveryManager->saveSettings(
                    $this->deliverySettingsNormalizer->mapToEntity($postSettings)
                        ->setDeliveryGateways($settings->getDeliveryGateways() ?? [])
                );

                $this->setSession('success', self::PAYSERA_SUCCESS);
            } else {
                $this->setSession('error_warning', self::PAYSERA_ERROR);
            }

            $this->response->redirect($this->getUrl(self::PAYSERA_EXTENSION_DELIVERY));
        }
        $tabs = [
            'main_settings' => $this->generateData(
                    'tab_main',
                    self::PAYSERA_EXTENSION_DELIVERY,
                    ['tab' => 'main_settings']
                ) + ['layout' => self::PAYSERA_EXTENSION_DELIVERY_TAB_MAIN_SETTINGS],
        ];

        if (
            $this->config->get(PayseraDeliverySettings::PROJECT_ID)
            && $this->config->get(PayseraDeliverySettings::PROJECT_PASSWORD)
        ) {
            $tabs[self::PAYSERA_DELIVERY_GATEWAY_LIST] = $this->generateData(
                    'tab_delivery_gateway',
                    self::PAYSERA_EXTENSION_DELIVERY,
                    ['tab' => self::PAYSERA_DELIVERY_GATEWAY_LIST]
                ) + ['layout' => self::PAYSERA_EXTENSION_DELIVERY_TAB_GATEWAY_LIST];

            foreach ($this->delivery->getDeliveryGateways() as $deliveryGateway) {
                if ($deliveryGateway->isEnabled() === false) {
                    continue;
                }

                $data[self::PAYSERA_DELIVERY_GATEWAY_LIST][] = $this->getDeliveryGateway($deliveryGateway);
            }
        }

        $data['content_title'] = $this->getText('content_title');

        foreach ($this->getErrorFieldName() as $fieldName) {
            $dataName = self::PAYSERA_ERROR_PREFIX . $fieldName;
            $data[$dataName] = $this->errorValue($fieldName);
        }

        foreach ($this->getSessionFieldName() as $sessionFieldName) {
            $dataName = self::PAYSERA_SESSION_PREFIX . $sessionFieldName;
            $data[$dataName] = $this->sessionValue($sessionFieldName);
        }

        foreach ($this->getBreadcrumbFields() as $key => $value) {
            $data['breadcrumbs'][] = $this->generateData($key, $value);
        }

        $data['action'] = $this->getUrl(self::PAYSERA_EXTENSION_DELIVERY);

        foreach ($this->getPayseraFieldsName() as $fieldName) {
            $data[$fieldName] = $this->generateConfigField($fieldName);
        }

        $data['currency'] = $this->model_localisation_currency->getCurrencyByCode(
            $this->config->get('config_currency')
        );
        $data['tabs'] = $tabs;
        $data['active_tab'] = $this->request->get['tab'] ?? 'main_settings';

        if ($data['active_tab'] === 'main_settings') {
            $this->load->model('localisation/order_status');

            $data['order_statuses'] = $this->model_localisation_order_status->getOrderStatuses();
        }

        $data['tab_content'] = $this->load->view($tabs[$data['active_tab']]['layout'], $data);

        $this->responseOutput(self::PAYSERA_EXTENSION_DELIVERY_VIEW, $data);
    }

    public function addRemoveDeliveryGateway()
    {
        $code = $this->request->get[self::PAYSERA_PARAM_CODE];
        $action = $this->request->get[self::PAYSERA_PARAM_ACTION] ?? null;
        $deliveryGateway = $this->delivery->getDeliveryGatewayByCode($code);

        if ($deliveryGateway !== null && $action !== null) {
            if ($action === self::PAYSERA_ACTION_ADD) {
                $this->deliveryManager->addGateway($code)->saveSettings();
            } else {
                $this->deliveryManager->removeGateway($code)->saveSettings();
            }

            $this->setSession('success', self::PAYSERA_SUCCESS);

            $this->response->redirect(
                $this->getUrl(
                    self::PAYSERA_EXTENSION_DELIVERY,
                    ['tab' => self::PAYSERA_DELIVERY_GATEWAY_LIST]
                )
            );
        }

        $this->errorNotFound();
    }

    public function shippingMethods()
    {
        $this->deliveryManager->syncShipmentMethods($this->delivery->getShippingMethods());

        $this->load->language(PayseraDeliverySettings::LANGUAGE_DELIVERY_GATEWAY);

        $code = $this->request->get[self::PAYSERA_PARAM_CODE];
        $action = $this->request->get[self::PAYSERA_PARAM_ACTION] ?? null;
        $key = $this->request->get[self::PAYSERA_PARAM_KEY] ?? null;
        $deliveryGateway = $this->delivery->getDeliveryGatewayByCode($code);

        if ($deliveryGateway === null) {
            return $this->errorNotFound();
        }

        if ($action !== null && $action === self::PAYSERA_ACTION_REMOVE && $key !== null) {
            $this->deliveryManager->removeGatewaySettings($code, $key)->saveSettings();

            $this->response->redirect(
                $this->getUrl(
                    $this->routePathResolver->getCompatibleRoutePath(
                        self::PAYSERA_EXTENSION_SHIPPING_METHODS_ROUTE_ACTION
                    ),
                    [self::PAYSERA_PARAM_CODE => $code]
                )
            );
        }

        foreach ($this->deliveryManager->getGatewayByCode($code) as $index => $deliveryGatewaySetting) {
            $data['shipping_methods'][] = $this->getShipmentGateway($deliveryGatewaySetting, $index);
        }

        $data['cancel'] = $this->getUrl(
            self::PAYSERA_EXTENSION_DELIVERY,
            ['tab' => self::PAYSERA_DELIVERY_GATEWAY_LIST]
        );
        $data['action'] = $this->getUrl(
            $this->routePathResolver->getCompatibleRoutePath(
                self::PAYSERA_EXTENSION_DELIVERY_GATEWAY_ROUTE_ACTION
            ),
            [
                self::PAYSERA_PARAM_CODE => $deliveryGateway->getCode(),
                self::PAYSERA_PARAM_ACTION => self::PAYSERA_ACTION_ADD,
            ]
        );

        $data['receiverTypes'] = $this->shipmentGatewayManager->getReceiverTypeOptions($deliveryGateway);

        return $this->responseOutput(self::PAYSERA_EXTENSION_SHIPPING_METHODS_VEW, $data);
    }

    public function about()
    {
        $this->load->language(PayseraDeliverySettings::LANGUAGE_ABOUT);

        $this->document->setTitle($this->getText('heading_title'));

        $data['text_about_content_updated'] = sprintf(
            $this->getText('text_about_content'),
            $this->getLink($this->getText('link_paysera_account'), 'Paysera')
        );
        $data['text_need_assistance_content_languages_updated'] = sprintf(
            $this->getText('text_need_assistance_content_languages'),
            $this->getLink($this->getText('link_contactus'), $this->getText('text_contactus'))
        );
        $data['text_need_assistance_content_socialmedia_updated'] = sprintf(
            $this->getText('text_need_assistance_content_socialmedia'),
            $this->getLink($this->getText('link_facebook'), 'Facebook'),
            $this->getLink($this->getText('link_twitter'), 'Twitter')
        );
        $data['text_explore_content_listitem_3_updated'] = sprintf(
            $this->getText('text_explore_content_listitem_3'),
            $this->getLink($this->getText('link_iban_account'), 'IBAN')
        );
        $data['text_explore_content_listitem_4_updated'] = sprintf(
            $this->getText('text_explore_content_listitem_4'),
            $this->getLink($this->getText('link_cart_visa'), $this->getText('text_visa')),
            $this->getLink($this->getText('link_google_pay'), 'Google Pay'),
            $this->getLink($this->getText('link_apple_pay'), 'Apple Pay')
        );
        $data['text_explore_content_outro_updated'] = sprintf(
            $this->getText('text_explore_content_outro'),
            $this->getLink($this->getText('link_mobile_application'), $this->getText('text_mobile_app'))
        );

        $this->responseOutput(self::PAYSERA_EXTENSION_ABOUT_VIEW, $data);
    }

    public function no_payment_plugin()
    {
        $this->load->language(PayseraDeliverySettings::LANGUAGE_NO_PLUGIN);

        $this->document->setTitle($this->getText('heading_title'));

        $data['paysera_payment_extension_link'] = sprintf(
            $this->getText('text_goto'),
            self::PAYSERA_EXTENSION_PAYMENT_LINK
        );

        $this->responseOutput(self::PAYSERA_EXTENSION_NO_PAYMENT_PLUGIN_VIEW, $data);
    }

    public function delivery_gateway()
    {
        $this->load->language(PayseraDeliverySettings::LANGUAGE_DELIVERY_GATEWAY);

        $this->document->setTitle($this->getText('heading_title'));

        $this->load->model('localisation/currency');
        $this->load->model('localisation/geo_zone');

        $code = $this->request->get[self::PAYSERA_PARAM_CODE];
        $key = $this->request->get[self::PAYSERA_PARAM_KEY] ?? null;
        $action = $this->request->get[self::PAYSERA_PARAM_ACTION] ?? null;
        $deliveryGateway = $this->delivery->getDeliveryGatewayByCode($code);
        $shippingMethodsUrl = $this->getUrl(
            $this->routePathResolver->getCompatibleRoutePath(
                self::PAYSERA_EXTENSION_SHIPPING_METHODS_ROUTE_ACTION
            ),
            [self::PAYSERA_PARAM_CODE => $code]
        );

        if (!$this->deliveryManager->isGatewayCodeExists($code) || $deliveryGateway === null) {
            return $this->errorNotFound();
        }

        $deliveryGatewayValidator = new DeliveryGatewayValidator(new MessageRepository($this->registry));

        $postSettings = $this->request->post[PayseraDeliverySettings::DELIVERY_GATEWAY] ?? [];
        if (count($postSettings)) {
            $deliveryGatewaySetting = $this->deliveryGatewaySettingsNormalizer->mapToEntity($postSettings);
            $deliveryGatewaySetting
                ->setShipmentGateway($deliveryGateway)
                ->setDescription(
                    $this->shipmentGatewayManager->getShipmentGatewayDescription(
                        $deliveryGatewaySetting->getReceiverType(),
                        $deliveryGateway
                    )
                )
            ;
        } else {
            $deliveryGatewaySetting = $this->deliveryManager->getGatewaySettings($code, $key);
        }

        if ($this->request->server['REQUEST_METHOD'] === self::REQUEST_METHOD_TYPE
            && $deliveryGatewayValidator->validate($postSettings)
        ) {
            if ($action === self::PAYSERA_ACTION_ADD) {
                $deliveryGatewaySetting->setTitle(
                    $postSettings[PayseraDeliverySettings::TITLE] ?? $deliveryGatewaySetting->getDescription()
                );

                if (
                    !$this->shipmentGatewayManager->isShippingMethodEnabled(
                        $this->deliveryHelper->resolveShippingMethodCode($deliveryGatewaySetting)
                    )
                ) {
                    $deliveryGatewaySetting->setSenderType(PayseraDeliverySettings::TYPE_PARCEL_MACHINE);
                }
            }

            $this->deliveryManager->updateGatewaySettings($code, $deliveryGatewaySetting, $key);

            $this->setSession('success', self::PAYSERA_SUCCESS);

            if ($action !== null || $key !== null) {
                $this->response->redirect($shippingMethodsUrl);
            }

            $this->response->redirect($this->getUrl(
                self::PAYSERA_EXTENSION_DELIVERY,
                ['tab' => self::PAYSERA_DELIVERY_GATEWAY_LIST]
            ));
        }

        $errors = $deliveryGatewayValidator->getProcessedErrors('error_');

        if ($deliveryGatewaySetting === null) {
            return $this->errorNotFound();
        }

        $currency = $this->model_localisation_currency->getCurrencyByCode(
            $this->config->get('config_currency')
        );
        $weightCode = $this->weight->getUnit($this->config->get('config_weight_class_id'));

        $data = [
            'label_free_delivery_threshold' => sprintf($this->getText('label_free_delivery'), $currency['code']),
            'delivery_gateway' => $this->deliveryGatewaySettingsNormalizer->mapFromEntity($deliveryGatewaySetting),
            'sender_types' => $this->shipmentGatewayManager->getSenderTypeOptions($deliveryGatewaySetting),
            'cancel' => $shippingMethodsUrl,
            'label_fee' => sprintf($this->getText('label_fee'), $currency['code']),
            'label_max_weight' => sprintf($this->getText('label_max_weight'), $weightCode),
            'label_min_weight' => sprintf($this->getText('label_min_weight'), $weightCode),
            'geo_zones' => $this->model_localisation_geo_zone->getGeoZones(),
        ];

        $data = array_merge($data, $errors);

        $this->responseOutput(self::PAYSERA_EXTENSION_DELIVERY_GATEWAY_VIEW, $data);
    }

    public function payseraDeliveryMenuHandler(string $eventRoute = '', array &$data = []): void
    {
        $this->load->language(PayseraDeliverySettings::LANGUAGE_DELIVERY_GATEWAY);

        $payseraMenu = array_filter($data['menus'], function ($menu) {
            return $menu['id'] === 'menu-paysera';
        });

        if (!count($payseraMenu)) {
            $data['menus'][] = [
                'id' => 'menu-paysera',
                'name' => sprintf(
                    $this->getText('menu_parent'),
                    '<img src="/extension/paysera_delivery/admin/view/image/shipping/paysera_menu_icon.svg" alt="paysera-logo" style="display: inline-block; margin-left: -6px; margin-right: 6px; width: 18px;"/>'
                ),
                'href' => '',
                'children' => [
                    'menu-paysera-about' => [
                        'name' => $this->getText('menu_about'),
                        'href' => $this->getUrl(
                            $this->routePathResolver->getCompatibleRoutePath(
                                self::PAYSERA_EXTENSION_ABOUT_ROUTE_ACTION
                            )
                        ),
                        'children' => [],
                    ],
                    'menu-paysera-delivery-settings' => [
                        'name' => $this->getText('menu_delivery_settings'),
                        'href' => $this->getUrl(self::PAYSERA_EXTENSION_DELIVERY),
                        'children' => [],
                    ],
                    'menu-paysera-payment-settings' => [
                        'name' => $this->getText('menu_payment_settings'),
                        'href' => $this->getUrl(
                            $this->routePathResolver->getCompatibleRoutePath(
                                self::PAYSERA_EXTENSION_NO_PAYMENT_PLUGIN_ROUTE_ACTION
                            )
                        ),
                        'children' => [],
                    ],
                ],
            ];
        } else {
            $payseraMenuIndex = array_key_first($payseraMenu);
            $data['menus'][$payseraMenuIndex]['children']['menu-paysera-delivery-settings'] = [
                'name' => $this->getText('menu_delivery_settings'),
                'href' => $this->getUrl(self::PAYSERA_EXTENSION_DELIVERY),
                'children' => [],
            ];
        }
    }

    public function install()
    {
        $this->load->model('setting/event');

        $events = [
            [
                'code'        => self::PAYSERA_EVENT_MENU_NAME,
                'trigger'     => self::PAYSERA_EVENT_MENU,
                'action'      => $this->routePathResolver->getCompatibleRoutePath(
                    self::PAYSERA_EXTENSION_MENU_ROUTE_ACTION
                ),
                'description' => '',
                'status'      => true,
                'sort_order'  => 1
            ],
            [
                'code'        => self::PAYSERA_DELIVERY_SHIPPING_METHOD_NAME,
                'trigger'     => self::PAYSERA_DELIVERY_EVENT_SHIPPING_METHOD,
                'action'      => $this->routePathResolver->getCompatibleRoutePath(
                    self::PAYSERA_EXTENSION_SHIPPING_METHODS_ROUTE_ACTION
                ),
                'description' => '',
                'status'      => true,
                'sort_order'  => 1
            ],
            [
                'code'        => self::PAYSERA_DELIVERY_CREATE_ORDER_NAME,
                'trigger'     => self::PAYSERA_DELIVERY_EVENT_ADD_ORDER,
                'action'      => $this->routePathResolver->getCompatibleRoutePath(
                    self::PAYSERA_EXTENSION_CREATE_DELIVERY_ROUTE_ACTION
                ),
                'description' => '',
                'status'      => true,
                'sort_order'  => 1
            ],
            [
                'code'        => self::PAYSERA_SHIPPING_METHOD_SAVE_NAME,
                'trigger'     => self::PAYSERA_SHIPPING_METHOD_SAVE,
                'action'      => $this->routePathResolver->getCompatibleRoutePath(
                    self::PAYSERA_SHIPPING_METHOD_SAVE_ROUTE_ACTION
                ),
                'description' => '',
                'status'      => true,
                'sort_order'  => 1
            ],
        ];

        foreach ($events as $event) {
            $this->model_setting_event->deleteEventByCode($event['code']);

            if (version_compare(VERSION, '4.0.0.0', '==')) {
                $this->model_setting_event->addEvent(
                    $event['code'],
                    $event['description'],
                    $event['trigger'],
                    $event['action'],
                    $event['status'],
                    $event['sort_order']
                );
            } else {
                $this->model_setting_event->addEvent($event);
            }
        }

        $this->load->model(self::PAYSERA_EXTENSION_DELIVERY);
		$this->model_extension_paysera_delivery_shipping_paysera->createTables();
    }

    public function uninstall()
    {
		$this->load->model('setting/event');
        $this->model_setting_event->deleteEventByCode(self::PAYSERA_EVENT_MENU_NAME);
        $this->model_setting_event->deleteEventByCode(self::PAYSERA_DELIVERY_SHIPPING_METHOD_NAME);
        $this->model_setting_event->deleteEventByCode(self::PAYSERA_DELIVERY_CREATE_ORDER_NAME);
        $this->model_setting_event->deleteEventByCode(self::PAYSERA_SHIPPING_METHOD_SAVE_NAME);

        $this->load->model(self::PAYSERA_EXTENSION_DELIVERY);
		$this->model_extension_paysera_delivery_shipping_paysera->deleteTables();
    }

    /**
     * @return bool
     */
    protected function validate()
    {
        if (!$this->user->hasPermission('modify', self::PAYSERA_EXTENSION_DELIVERY)) {
            $this->error['warning'] = $this->getText('error_warning');
        }

        if (!$this->request->post[PayseraDeliverySettings::PROJECT_ID]) {
            $this->error['project_id'] = $this->getText('error_project');
        }

        if (!$this->request->post[PayseraDeliverySettings::PROJECT_PASSWORD]) {
            $this->error['project_password'] = $this->getText('error_sign');
        }

        return !$this->error;
    }

    /**
     * @param string $fieldName
     * @return string|int|bool
     */
    private function generateConfigField($fieldName)
    {
        if (isset($this->request->post[$fieldName])) {
            $data = $this->request->post[$fieldName];
        } else {
            $data = $this->config->get($fieldName);
        }

        return $data;
    }

    /**
     * @return string[]
     */
    private function getErrorFieldName()
    {
        return $this->errorFieldName;
    }

    /**
     * @return string[]
     */
    private function getSessionFieldName()
    {
        return $this->sessionFieldName;
    }

    /**
     * @return array
     */
    private function getPayseraFieldsName()
    {
        return $this->payseraFieldsName;
    }

    /**
     * @return string[]
     */
    public function getBreadcrumbFields()
    {
        return $this->breadcrumbFields;
    }

    /**
     * @param string $fieldName
     * @return string
     */
    private function errorValue($fieldName)
    {
        if (isset($this->error[$fieldName])) {
            $data = $this->error[$fieldName];
        } else {
            $data = self::PAYSERA_EMPTY_VALUE;
        }

        return $data;
    }

    /**
     * @param string $sessionKey
     * @return string
     */
    private function sessionValue($sessionKey)
    {
        if (isset($this->session->data[$sessionKey])) {
            $data = $this->session->data[$sessionKey];

            unset($this->session->data[$sessionKey]);
        } else {
            $data = self::PAYSERA_EMPTY_VALUE;
        }

        return $data;
    }

    /**
     * @param ShipmentGateway $shipmentGateway
     * @return array
     */
    private function getDeliveryGateway($shipmentGateway)
    {
        $isAdded = $this->deliveryManager->isGatewayCodeExists($shipmentGateway->getCode());
        $shippingMethods = [];

        foreach ($this->deliveryManager->getGatewayByCode($shipmentGateway->getCode()) as $deliveryGatewaySettings) {
            $shippingMethods[] = array_merge(
                $this->deliveryGatewaySettingsNormalizer->mapFromEntity($deliveryGatewaySettings),
                [
                    'shipping_method' => $deliveryGatewaySettings->getTitle()
                ]
            );
        }

        return array_merge(
            $this->shipmentGatewayNormalizer->mapFromEntity($shipmentGateway),
            [
                'shipping_methods' => $shippingMethods,
                'edit_url' => $this->getUrl(
                    $this->routePathResolver->getCompatibleRoutePath(
                        self::PAYSERA_EXTENSION_SHIPPING_METHODS_ROUTE_ACTION
                    ),
                    [self::PAYSERA_PARAM_CODE => $shipmentGateway->getCode()]
                ),
                'add_remove_url' => $this->getUrl(
                    $this->routePathResolver->getCompatibleRoutePath(
                        self::PAYSERA_EXTENSION_DELIVERY_GATEWAY_ADD_REMOVE_ROUTE_ACTION
                    ),
                    [
                        self::PAYSERA_PARAM_CODE => $shipmentGateway->getCode(),
                        self::PAYSERA_PARAM_ACTION => $isAdded ? self::PAYSERA_ACTION_REMOVE : self::PAYSERA_ACTION_ADD,
                    ]
                ),
                'is_added' => $isAdded,
            ]
        );
    }

    /**
     * @param DeliveryGatewaySettings $deliveryGatewaySettings
     * @param int $index
     * @return array
     */
    private function getShipmentGateway($deliveryGatewaySettings, $index)
    {
        return array_merge(
            $this->deliveryGatewaySettingsNormalizer->mapFromEntity($deliveryGatewaySettings),
            [
                'sender' => $this->shipmentGatewayManager
                    ->getShipmentGatewayDescription($deliveryGatewaySettings->getSenderType())
                ,
                'remove_url' => $this->getUrl(
                    $this->routePathResolver->getCompatibleRoutePath(
                        self::PAYSERA_EXTENSION_SHIPPING_METHODS_ROUTE_ACTION
                    ),
                    [
                        self::PAYSERA_PARAM_CODE => $deliveryGatewaySettings->getShipmentGateway()->getCode(),
                        self::PAYSERA_PARAM_KEY => $index,
                        self::PAYSERA_PARAM_ACTION => self::PAYSERA_ACTION_REMOVE
                    ]
                ),
                'edit_url' => $this->getUrl(
                    $this->routePathResolver->getCompatibleRoutePath(
                        self::PAYSERA_EXTENSION_DELIVERY_GATEWAY_ROUTE_ACTION
                    ),
                    [
                        self::PAYSERA_PARAM_CODE => $deliveryGatewaySettings->getShipmentGateway()->getCode(),
                        self::PAYSERA_PARAM_KEY => $index,
                    ]
                ),
            ]
        );
    }

    private function errorNotFound()
    {
        $this->load->language('error/not_found');

        $this->document->setTitle($this->getText('heading_title'));

        $data['breadcrumbs'] = [
            $this->generateData('text_home', 'common/dashboard'),
            $this->generateData('heading_title', 'error/not_found'),
        ];

        $data['header'] = $this->load->controller('common/header');
        $data['column_left'] = $this->load->controller('common/column_left');
        $data['footer'] = $this->load->controller('common/footer');

        return $this->response->setOutput($this->load->view('error/not_found', $data));
    }

    /**
     * @param string $text
     * @param string $path
     * @param array $additionalParams
     * @return array
     */
    private function generateData($text, $path, $additionalParams = [])
    {
        $httpBuildQuery = http_build_query(
            array_merge(
                [
                    self::PAYSERA_PARAM_USER_TOKEN => $this->session->data[self::PAYSERA_PARAM_USER_TOKEN],
                ],
                $additionalParams
            )
        );

        if (empty($text)) {
            $data = $this->url->link($path, $httpBuildQuery, true);
        } elseif (empty($path)) {
            $data = $this->getText($text);
        } else {
            $data = [
                'text' => $this->getText($text),
                'href' => $this->url->link($path, $httpBuildQuery, true),
            ];
        }

        return $data;
    }

    /**
     * @param string $name
     * @param string $value
     */
    private function setSession($name, $value)
    {
        $this->session->data[$name] = $this->generateData($value, self::PAYSERA_EMPTY_VALUE);
    }

    /**
     * @param string $route
     * @param array $args
     * @return array
     */
    private function getUrl($route, $args = [])
    {
        return $this->generateData(self::PAYSERA_EMPTY_VALUE, $route, $args);
    }

    /**
     * @param string $translationKey
     * @return string
     */
    private function getText($translationKey)
    {
        return $this->language->get($translationKey);
    }

    /**
     * @param string $link
     * @param string $text
     * @return string
     */
    private function getLink($link, $text)
    {
        return sprintf('<a href="%s" target="_blank" rel="noopener noreferrer">%s</a>', $link, $text);
    }

    /**
     * @param string $template
     * @param array $data
     * @return string
     */
    private function responseOutput($template, $data)
    {
        $data['header'] = $this->load->controller('common/header');
        $data['column_left'] = $this->load->controller('common/column_left');
        $data['footer'] = $this->load->controller('common/footer');

        return $this->response->setOutput($this->load->view($template, $data));
    }
}
