<?php

declare(strict_types=1);

namespace Paysera\Delivery\ValueObject;

class OpenCartVersion
{
    private string $version;

    public function __construct(string $version)
    {
        $this->validateVersion($version);
        $this->version = $version;
    }

    public static function fromString(string $version): self
    {
        return new self($version);
    }

    public function toString(): string
    {
        return $this->version;
    }

    public function __toString(): string
    {
        return $this->version;
    }

    public function equals(OpenCartVersion $other): bool
    {
        return $this->version === $other->version;
    }

    public function isGreaterThan(OpenCartVersion $other): bool
    {
        return version_compare($this->version, $other->version, '>');
    }

    public function isGreaterThanOrEqual(OpenCartVersion $other): bool
    {
        return version_compare($this->version, $other->version, '>=');
    }

    public function isLessThan(OpenCartVersion $other): bool
    {
        return version_compare($this->version, $other->version, '<');
    }

    public function isLessThanOrEqual(OpenCartVersion $other): bool
    {
        return version_compare($this->version, $other->version, '<=');
    }

    private function validateVersion(string $version): void
    {
        if (empty($version)) {
            throw new \InvalidArgumentException('Version cannot be empty');
        }

        if (!preg_match('/^\d+\.\d+\.\d+\.\d+$/', $version)) {
            throw new \InvalidArgumentException(
                sprintf('Invalid version format: %s. Expected format: x.y.z.w', $version)
            );
        }
    }
}
