<?php

declare(strict_types=1);

namespace Paysera\Delivery\ValueObject;

/**
 * Value Object representing the Paysera Delivery extension version
 *
 * Ensures type safety and validation for extension version strings
 */
class ExtensionVersion
{
    private string $version;

    /**
     * @param string $version Extension version string (e.g., '2.1.0', '2.0.0')
     * @throws \InvalidArgumentException If version format is invalid
     */
    public function __construct(string $version)
    {
        $this->validateVersion($version);
        $this->version = $version;
    }

    /**
     * Check if this version equals another version
     */
    public function equals(ExtensionVersion $other): bool
    {
        return $this->version === $other->version;
    }

    /**
     * Check if this version is greater than another version
     */
    public function isGreaterThan(ExtensionVersion $other): bool
    {
        return version_compare($this->version, $other->version, '>');
    }

    /**
     * Check if this version is greater than or equal to another version
     */
    public function isGreaterThanOrEqual(ExtensionVersion $other): bool
    {
        return version_compare($this->version, $other->version, '>=');
    }

    /**
     * Check if this version is less than another version
     */
    public function isLessThan(ExtensionVersion $other): bool
    {
        return version_compare($this->version, $other->version, '<');
    }

    /**
     * Get the version string
     */
    public function toString(): string
    {
        return $this->version;
    }

    /**
     * Allow implicit string conversion
     */
    public function __toString(): string
    {
        return $this->version;
    }

    /**
     * Validate version format
     *
     * @throws \InvalidArgumentException If version format is invalid
     */
    private function validateVersion(string $version): void
    {
        // Semantic versioning format: MAJOR.MINOR.PATCH (e.g., 2.1.0)
        if (!preg_match('/^\d+\.\d+\.\d+$/', $version)) {
            throw new \InvalidArgumentException(
                sprintf('Invalid extension version format: %s. Expected format: x.y.z', $version)
            );
        }
    }
}
