<?php

declare(strict_types=1);

namespace Paysera\Delivery\Update;

use Throwable;
use Paysera\Delivery\DatabaseTable;
use Paysera\Delivery\Service\CompatibilityService\PayseraCompatibilityService;
use Paysera\Delivery\Provider\EventsProvider;
use Paysera\Delivery\Service\Logger;
use Paysera\Delivery\Exception\MigrationException;
use Paysera\Delivery\Exception\MigrationApplyException;
use Paysera\Delivery\Exception\MigrationRollbackException;
use Paysera\Delivery\PayseraDeliverySettings;
use Paysera\Delivery\ValueObject\ExtensionVersion;

class UpdateManager
{
    const INITIAL_UPDATE_VERSION = '2.0.1';

    private $modelSetting;
    private $db;
    private $modelSettingEvent;
    private PayseraCompatibilityService $compatibilityService;
    private EventsProvider $eventsProvider;
    private Logger $logger;
    private ?ExtensionVersion $installedVersion;

    public function __construct(
        $modelSetting,
        $db,
        $modelSettingEvent,
        PayseraCompatibilityService $compatibilityService,
        EventsProvider $eventsProvider,
        Logger $logger
    ) {
        $this->modelSetting = $modelSetting;
        $this->db = $db;
        $this->modelSettingEvent = $modelSettingEvent;
        $this->compatibilityService = $compatibilityService;
        $this->eventsProvider = $eventsProvider;
        $this->logger = $logger;

        $settings = $this->modelSetting->getSetting(PayseraDeliverySettings::SETTINGS_EXTENSION_STATUS_NAME);
        $versionString = $settings[PayseraDeliverySettings::EXTENSION_VERSION_KEY] ?? null;
        $this->installedVersion = $versionString ? new ExtensionVersion($versionString) : null;
    }

    /**
     * @throws MigrationApplyException
     * @throws MigrationRollbackException
     */
    public function applyUpdates(): void
    {
        if (!$this->isInstalled()) {
            return;
        }

        if ($this->installedVersion === null) {
            $this->logger->log('Plugin installed but version not set, initializing to version ' . self::INITIAL_UPDATE_VERSION);
            $this->initializeVersion();
        }

        $currentVersion = $this->installedVersion;
        $migrations = include __DIR__ . '/../config/migrations.php';

        if ($this->isUpdateNeeded($currentVersion, $migrations)) {
            $this->logger->log('Update needed from version ' . $currentVersion->toString());

            foreach ($migrations as $version => $migrationClass) {
                $migrationVersion = new ExtensionVersion($version);
                if ($currentVersion->isLessThan($migrationVersion)) {
                    $this->logger->log('Applying migration ' . $version);
                    $this->applyMigration($migrationClass);
                    $this->logger->log('Successfully applied migration ' . $version);
                }
            }

            $this->logger->log('All migrations completed successfully');
        }
    }

    private function isInstalled(): bool
    {
        if ($this->installedVersion !== null) {
            return true;
        }

        try {
            $tableName = $this->db->escape(DatabaseTable::SHIPPING_REQUEST);
            $result = $this->db->query('SELECT 1 FROM `' . $tableName . '` LIMIT 1');
            return !is_bool($result);
        } catch (Throwable $exception) {
            return false;
        }
    }

    /**
     * @throws MigrationException
     */
    private function initializeVersion(): void
    {
        try {
            $version = new ExtensionVersion(self::INITIAL_UPDATE_VERSION);
            $this->logger->log('Initializing version to ' . $version->toString());
            $this->updateVersion($version);
        } catch (Throwable $exception) {
            $message = 'Failed to initialize version: ' . $exception->getMessage();
            $this->logger->log($message);
            throw new MigrationException($message, 0, $exception);
        }
    }

    private function isUpdateNeeded(ExtensionVersion $currentVersion, array $migrations): bool
    {
        if (count($migrations) === 0) {
            return false;
        }

        $latestVersionString = array_key_last($migrations);
        $latestVersion = new ExtensionVersion($latestVersionString);

        return $currentVersion->isLessThan($latestVersion);
    }

    /**
     * @throws MigrationApplyException
     * @throws MigrationRollbackException
     */
    private function applyMigration(string $migrationClass): void
    {
        $migration = new $migrationClass(
            $this->db,
            $this->modelSettingEvent,
            $this->compatibilityService,
            $this->eventsProvider
        );

        $versionString = $migration->getVersion();
        $version = new ExtensionVersion($versionString);

        try {
            $this->db->query('START TRANSACTION');
            $this->logger->log('Transaction started for migration ' . $versionString);

            $migration->apply();
            $this->updateVersion($version);

            $this->db->query('COMMIT');
            $this->logger->log('Transaction committed for migration ' . $versionString);
        } catch (Throwable $exception) {
            $errorMessage = 'Migration ' . $versionString . ' failed: ' . $exception->getMessage();
            $this->logger->log($errorMessage);

            try {
                $this->db->query('ROLLBACK');
                $this->logger->log('Transaction rolled back for migration ' . $versionString);

                throw new MigrationApplyException(
                    $errorMessage . '. Transaction has been rolled back.',
                    0,
                    $exception
                );
            } catch (Throwable $rollbackException) {
                $rollbackMessage = 'Failed to rollback migration ' . $versionString . ': ' . $rollbackException->getMessage();
                $this->logger->log($rollbackMessage);

                throw new MigrationRollbackException(
                    $rollbackMessage . ' Original error: ' . $exception->getMessage(),
                    0,
                    $rollbackException
                );
            }
        }
    }

    /**
     * @throws MigrationException
     */
    private function updateVersion(ExtensionVersion $version): void
    {
        try {
            $versionString = $version->toString();
            $this->modelSetting->editSetting(PayseraDeliverySettings::SETTINGS_EXTENSION_STATUS_NAME, [
                PayseraDeliverySettings::EXTENSION_VERSION_KEY => $versionString,
            ]);

            $this->installedVersion = $version;
            $this->logger->log('Version updated to ' . $versionString);
        } catch (Throwable $exception) {
            $message = 'Failed to update version to ' . $version->toString() . ': ' . $exception->getMessage();
            $this->logger->log($message);
            throw new MigrationException($message, 0, $exception);
        }
    }
}
