<?php

declare(strict_types=1);

namespace Paysera\Delivery\Update\Update;

use RuntimeException;
use Throwable;
use Paysera\Delivery\DatabaseTable;
use Paysera\Delivery\Update\MigrationInterface;
use Paysera\Delivery\Service\CompatibilityService\PayseraCompatibilityService;
use Paysera\Delivery\Provider\EventsProvider;

class Update_2_1_0 implements MigrationInterface
{
    private const MIGRATION_VERSION = '2.1.0';

    private $db;
    private $modelSettingEvent;
    private PayseraCompatibilityService $compatibilityService;
    private EventsProvider $eventsProvider;

    public function __construct(
        $db,
        $modelSettingEvent,
        PayseraCompatibilityService $compatibilityService,
        EventsProvider $eventsProvider
    ) {
        $this->db = $db;
        $this->modelSettingEvent = $modelSettingEvent;
        $this->compatibilityService = $compatibilityService;
        $this->eventsProvider = $eventsProvider;
    }

    public function apply(): void
    {
        $this->addDeliveryOrderIdColumn();
        $this->removeDeprecatedEvents();
        $this->registerNewEvents();
    }

    private function addDeliveryOrderIdColumn(): void
    {
        try {
            $checkQuery = sprintf(
                'SHOW COLUMNS FROM `%s` LIKE \'delivery_order_id\'',
                DatabaseTable::SHIPPING_REQUEST
            );

            $result = $this->db->query($checkQuery);

            if ($result->num_rows === 0) {
                $this->db->query(
                    'ALTER TABLE `' . DatabaseTable::SHIPPING_REQUEST .
                    '` ADD `delivery_order_id` VARCHAR(255) NULL AFTER `order_id`'
                );
            }
        } catch (Throwable $exception) {
            throw new RuntimeException(
                'Failed to add delivery_order_id column: ' . $exception->getMessage(),
                0,
                $exception
            );
        }
    }


    private function eventExists(string $code): bool
    {
        try {
            $query = sprintf(
                'SELECT COUNT(*) as count FROM `%sevent` WHERE `code` = \'%s\'',
                DB_PREFIX,
                $this->db->escape($code)
            );

            $result = $this->db->query($query);

            return $result->row['count'] > 0;
        } catch (Throwable $exception) {
            throw new RuntimeException(
                'Failed to check event existence: ' . $exception->getMessage(),
                0,
                $exception
            );
        }
    }

    /**
     * Remove deprecated events that were replaced in version 2.1.0
     */
    private function removeDeprecatedEvents(): void
    {
        $deprecatedEventCodes = [
            'paysera_delivery_checkout_get_total',
        ];

        $eventCodesToRemove = [];
        foreach ($deprecatedEventCodes as $code) {
            if ($this->eventExists($code)) {
                $eventCodesToRemove[] = $code;
            }
        }

        if (count($eventCodesToRemove) > 0) {
            $this->compatibilityService
                ->getCurrentOCVersionAdapter($this->modelSettingEvent)
                ->removeEvents($eventCodesToRemove)
            ;
        }
    }

    /**
     * Register new events introduced in version 2.1.0
     * Gets all events from config file and registers those that don't exist yet
     */
    private function registerNewEvents(): void
    {
        $allEvents = $this->eventsProvider->getEvents();
        $newEventCodes = [
            'paysera_delivery_payment_confirmed',
            'paysera_delivery_add_shipping_logo_before_checkout_total',
            'paysera_delivery_remove_shipping_logo_after_checkout_total',
        ];

        $eventsToRegister = [];

        foreach ($allEvents as $event) {
            if (in_array($event->getCode(), $newEventCodes, true) && !$this->eventExists($event->getCode())) {
                $eventsToRegister[] = $event;
            }
        }

        if (count($eventsToRegister) > 0) {
            $this->compatibilityService
                ->getCurrentOCVersionAdapter($this->modelSettingEvent)
                ->registerEvents($eventsToRegister)
            ;
        }
    }

    public function getVersion(): string
    {
        return self::MIGRATION_VERSION;
    }
}
