<?php

namespace Paysera\Delivery;

use Paysera\Delivery\Entity\DeliveryGatewaySettings;
use Paysera\Delivery\Normalizer\ShipmentMethodNormalizer;
use Paysera\DeliveryApi\MerchantClient\Entity\ShipmentGateway;

class ShipmentGatewayManager
{
    private const ALL_ZONES_GEOZONE_ID = 0;

    private $registry;
    private $deliveryLibraryHelper;
    private $shipmentMethodNormalizer;
    private $deliveryResolver;
    private $parcelDisabledGatewayCodes = ['tnt', 'itella'];

    public function __construct($registry, PayseraDeliveryLibraryHelper $deliveryLibraryHelper)
    {
        $this->registry = $registry;
        $this->deliveryLibraryHelper = $deliveryLibraryHelper;
        $this->shipmentMethodNormalizer = new ShipmentMethodNormalizer();
        $this->deliveryResolver = new PayseraDeliveryResolver();
    }

    public function __get($key)
    {
        return $this->registry->get($key);
    }

    /**
     * @param ShipmentGateway $shipmentGateway
     * @return array
     */
    public function getReceiverTypeOptions($shipmentGateway)
    {
        $this->load->language(PayseraDeliverySettings::LANGUAGE_SETTINGS);

        $receiverTypes = [];

        foreach ($this->getShipmentMethodGroupByReceiverCode() as $receiverType => $shipmentMethods) {
            $receiverTypes[$receiverType] = $this->getShipmentGatewayDescription($receiverType, $shipmentGateway);
        }

        if (in_array(strtolower($shipmentGateway['code'] ?? ''), $this->parcelDisabledGatewayCodes, true)) {
            unset($receiverTypes['parcel-machine']);
        }

        return $receiverTypes;
    }

    /**
     * @param DeliveryGatewaySettings $deliveryGatewaySettings
     * @return array
     */
    public function getSenderTypeOptions($deliveryGatewaySettings)
    {
        $this->load->language(PayseraDeliverySettings::LANGUAGE_SETTINGS);

        $shipmentMethods = $this->getShipmentMethodGroupByReceiverCode();
        $gatewayCode = strtolower($deliveryGatewaySettings->getShipmentGateway()->getCode());
        $senderTypes = [];

        if (isset($shipmentMethods[$deliveryGatewaySettings->getReceiverType()])) {
            foreach (PayseraDeliverySettings::AVAILABLE_TYPES as $type => $translationKey) {
                if ($type === PayseraDeliverySettings::TYPE_PARCEL_MACHINE
                    && in_array($gatewayCode, $this->parcelDisabledGatewayCodes, true)) {
                    continue;
                }

                $methodCode = $this->deliveryResolver->resolveShippingMethodCode(
                    (new DeliveryGatewaySettings())
                        ->setReceiverType($deliveryGatewaySettings->getReceiverType())
                        ->setSenderType($type)
                );

                if (in_array($methodCode, $shipmentMethods[$deliveryGatewaySettings->getReceiverType()], true)) {
                    $senderTypes[$type] = $this->getShipmentGatewayDescription($type);
                }
            }
        }

        return $senderTypes;
    }

    /**
     * @param string $shippingMethod
     * @return bool
     */
    public function isShippingMethodEnabled($shippingMethod)
    {
        $shipmentMethods = $this->shipmentMethodNormalizer->mapAssociativeArrayFromEntity(
            $this->deliveryLibraryHelper->getShippingMethods()
        );

        return isset($shipmentMethods[$shippingMethod]) && $shipmentMethods[$shippingMethod]['enabled'] === true;
    }

    /**
     * @param string $shippingMethodCode
     * @param ?ShipmentGateway $shipmentGateway
     * @return string
     */
    public function getShipmentGatewayDescription($shippingMethodCode, $shipmentGateway = null)
    {
        $shippingMethod = [];

        if ($shipmentGateway !== null) {
            $shippingMethod[] = $shipmentGateway->getDescription();
        }

        if ($shipmentGateway !== null && $shippingMethodCode === PayseraDeliverySettings::TYPE_PARCEL_MACHINE) {
            $shippingMethod[] = $this->language->get(PayseraDeliverySettings::LANGUAGE_TERMINAL_KEY);
        } else {
            $shippingMethod[] = $this->language->get(PayseraDeliverySettings::AVAILABLE_TYPES[$shippingMethodCode]);
        }

        return implode(' ', $shippingMethod);
    }

    /**
     * @return array
     */
    public function getShipmentMethodGroupByReceiverCode()
    {
        $shippingMethods = [];

        foreach ($this->deliveryLibraryHelper->getShippingMethods() as $shippingMethod) {
            if ($shippingMethod->isEnabled()) {
                $shippingMethods[$shippingMethod->getReceiverCode()][] = $shippingMethod->getCode();
            }
        }

        return $shippingMethods;
    }

    public function isShippingMethodInGeoZone(int $countryId, string $zones, ?int $geoZoneId): bool
    {
        if ($geoZoneId === null || $geoZoneId === self::ALL_ZONES_GEOZONE_ID) {
            return true;
        }

        $sanitizedZones = preg_replace('/[^0-9,]/', '', $zones);

        $geoZones = $this->db->query(
            sprintf('SELECT DISTINCT * FROM %szone_to_geo_zone '
                . ' WHERE 1 '
                . ' AND country_id = %d '
                . ' AND zone_id IN (%s) '
                . ';', DB_PREFIX, $countryId, $sanitizedZones));

        $geoZones = array_filter($geoZones->rows, function ($geoZone) use ($geoZoneId) {
            return ((int) $geoZone['geo_zone_id'] === (int) $geoZoneId);
        });

        return (bool) count($geoZones);
    }
}
