<?php

declare(strict_types=1);

namespace Paysera\Delivery\Service;

use Paysera\Delivery\Entity\Event;
use Paysera\Delivery\Service\CompatibilityService\PayseraCompatibilityService;
use Paysera\Delivery\Provider\EventsProvider;
use Paysera\Delivery\PayseraDeliverySettings;
use Paysera\Delivery\Exception\EventRetrievalException;
use Paysera\Delivery\Exception\StoredVersionUpdateException;
use Paysera\Delivery\Exception\EventReinstallationException;
use Paysera\Delivery\ValueObject\OpenCartVersion;
use Throwable;

class SelfCheckUpService
{
    private const PAYSERA_EVENT_PREFIX = 'paysera_delivery_';
    private OpenCartVersion $currentOpenCartVersion;
    private ?OpenCartVersion $installedOpenCartVersion;
    private $modelSettingSetting;
    private $modelSettingEvent;
    private PayseraCompatibilityService $compatibilityService;
    private EventsProvider $eventsProvider;
    private ?string $installedPluginVersion;

    public function __construct(
        OpenCartVersion $currentOpenCartVersion,
        ?OpenCartVersion $installedOpenCartVersion,
        $modelSettingSetting,
        $modelSettingEvent,
        PayseraCompatibilityService $compatibilityService,
        EventsProvider $eventsProvider,
        ?string $installedPluginVersion
    ) {
        $this->currentOpenCartVersion = $currentOpenCartVersion;
        $this->installedOpenCartVersion = $installedOpenCartVersion;
        $this->modelSettingSetting = $modelSettingSetting;
        $this->modelSettingEvent = $modelSettingEvent;
        $this->compatibilityService = $compatibilityService;
        $this->eventsProvider = $eventsProvider;
        $this->installedPluginVersion = $installedPluginVersion;
    }

    public function performCheckUp(): void
    {
        if ($this->shouldSkipCheckUp()) {
            return;
        }

        if ($this->needsEventReinstallation()) {
            $this->reinstallEventListeners();
            $this->updateStoredOpenCartVersion();
        }
    }

    private function shouldSkipCheckUp(): bool
    {
        return $this->installedPluginVersion === null ||
            ($this->installedOpenCartVersion !== null &&
                $this->installedOpenCartVersion->equals($this->currentOpenCartVersion));
    }

    private function needsEventReinstallation(): bool
    {
        return !$this->areEventsProperlyInstalled();
    }

    private function areEventsProperlyInstalled(): bool
    {
        $expectedEvents = $this->getExpectedEvents();
        $installedEvents = $this->getInstalledPayseraEvents();

        /** @var Event $expectedEvent */
        foreach ($expectedEvents as $expectedEvent) {
            $eventCode = $expectedEvent->getCode();

            if (!isset($installedEvents[$eventCode]) ||
                $installedEvents[$eventCode]['trigger'] !== $expectedEvent->getTrigger() ||
                !$installedEvents[$eventCode]['status']) {
                return false;
            }
        }

        return true;
    }

    /**
     * Get expected events from events provider
     *
     * @return Event[]
     */
    private function getExpectedEvents(): array
    {
        return $this->eventsProvider->getEvents();
    }

    /**
     * Get installed paysera events from model setting event
     *
     * @return Event[]
     */
    private function getInstalledPayseraEvents(): array
    {
        $events = [];
        try {
            $allEvents = $this->modelSettingEvent->getEvents();

            if (is_array($allEvents)) {
                foreach ($allEvents as $event) {
                    if (isset($event['code']) && $this->isPayseraEvent($event['code'])) {
                        $events[$event['code']] = $event;
                    }
                }
            }
        } catch (Throwable $e) {
            throw new EventRetrievalException('SelfCheckUpService::getInstalledPayseraEvents failed: ' . $e->getMessage(), 0, $e);
        }

        return $events;
    }

    /**
     * Check if event is paysera event
     *
     * @param string $eventCode
     * @return bool
     */
    private function isPayseraEvent(string $eventCode): bool
    {
        return strpos($eventCode, self::PAYSERA_EVENT_PREFIX) === 0;
    }

    /**
     * Update stored open cart version
     *
     * @return void
     */
    private function updateStoredOpenCartVersion(): void
    {
        try {
            $currentSettings = $this->modelSettingSetting->getSetting(PayseraDeliverySettings::SETTINGS_EXTENSION_STATUS_NAME) ?? [];
            $currentSettings[PayseraDeliverySettings::OPENCART_VERSION_KEY] = $this->currentOpenCartVersion->toString();

            $this->modelSettingSetting->editSetting(PayseraDeliverySettings::SETTINGS_EXTENSION_STATUS_NAME, $currentSettings);
        } catch (Throwable $e) {
            throw new StoredVersionUpdateException('SelfCheckUpService::updateStoredOpenCartVersion failed: ' . $e->getMessage(), 0, $e);
        }
    }

    private function reinstallEventListeners(): void
    {
        try {
            $events = $this->eventsProvider->getEvents();
            $eventCodes = array_map(fn($event) => $event->getCode(), $events);

            $adapter = $this->compatibilityService->getCurrentOCVersionAdapter($this->modelSettingEvent);

            $adapter->removeEvents($eventCodes);

            $adapter->registerEvents($events);

        } catch (Throwable $e) {
            throw new EventReinstallationException('Failed to reinstall event listeners: ' . $e->getMessage(), 0, $e);
        }
    }
}
