<?php

declare(strict_types=1);

namespace Paysera\Delivery\Service;

use Paysera\Delivery\Service\CompatibilityService\PayseraCompatibilityService;
use Paysera\Delivery\PayseraDeliverySettings;
use Paysera\Delivery\Provider\EventsProvider;
use Paysera\Delivery\ValueObject\ExtensionVersion;
use Paysera\Delivery\ValueObject\OpenCartVersion;

class PayseraDeliveryInstaller
{
    private PayseraCompatibilityService $compatibilityService;
    private $payseraDeliveryModel;
    private $modelSetting;
    private $modelSettingEvent;
    private ExtensionVersion $extensionVersion;
    private OpenCartVersion $currentOCVersion;
    private EventsProvider $eventsProvider;

    public function __construct(
        PayseraCompatibilityService $compatibilityService,
        $payseraDeliveryModel,
        $modelSetting,
        $modelSettingEvent,
        ExtensionVersion $extensionVersion,
        OpenCartVersion $currentOCVersion,
        EventsProvider $eventsProvider
    ) {
        $this->compatibilityService = $compatibilityService;
        $this->payseraDeliveryModel = $payseraDeliveryModel;
        $this->modelSetting = $modelSetting;
        $this->modelSettingEvent = $modelSettingEvent;
        $this->extensionVersion = $extensionVersion;
        $this->currentOCVersion = $currentOCVersion;
        $this->eventsProvider = $eventsProvider;
    }

    public function install(): void
    {
        $this->createTables();
        $this->registerEvents();
        $this->setVersionSetting($this->extensionVersion);
    }

    public function uninstall(): void
    {
        $this->removeEvents();
        $this->dropTables();
        $this->removeVersionSetting();
    }

    public function registerEvents(): void
    {
        $events = $this->eventsProvider->getEvents();
        $this->compatibilityService
            ->getCurrentOCVersionAdapter($this->modelSettingEvent)
            ->registerEvents($events);
    }

    public function createTables(): void
    {
        $this->payseraDeliveryModel->createTables();
    }

    public function removeEvents(): void
    {
        $events = $this->eventsProvider->getEvents();
        $eventCodes = array_map(fn($event) => $event->getCode(), $events);
        $this->compatibilityService
            ->getCurrentOCVersionAdapter($this->modelSettingEvent)
            ->removeEvents($eventCodes);
    }

    public function dropTables(): void
    {
        $this->payseraDeliveryModel->deleteTables();
    }

    public function setVersionSetting(ExtensionVersion $extensionVersion): void
    {
        $this->modelSetting->editSetting(PayseraDeliverySettings::SETTINGS_EXTENSION_STATUS_NAME, [
            PayseraDeliverySettings::EXTENSION_VERSION_KEY => $extensionVersion->toString(),
            PayseraDeliverySettings::OPENCART_VERSION_KEY => $this->currentOCVersion->toString(),
        ]);
    }

    public function removeVersionSetting(): void
    {
        $this->modelSetting->deleteSetting(PayseraDeliverySettings::SETTINGS_EXTENSION_STATUS_NAME);
        $this->modelSetting->deleteSetting(PayseraDeliverySettings::SETTINGS_NAME);
    }
}
