<?php

declare(strict_types=1);

namespace Paysera\Delivery\Service\Event;

use Paysera\Delivery\Service\Event\EventAdapter\OpenCartV4000EventAdapter;
use Paysera\Delivery\Service\Event\EventAdapter\OpenCartV4010EventAdapter;
use Paysera\Delivery\Service\Event\EventAdapter\OpenCartV4103EventAdapter;
use Paysera\Delivery\Exception\UnsupportedVersionException;


class EventAdapterFactory
{

    private const ADAPTER_CLASSES = [
        '4.0.0.0' => OpenCartV4000EventAdapter::class,
        '4.0.1.0' => OpenCartV4010EventAdapter::class,
        '4.1.0.3' => OpenCartV4103EventAdapter::class,
    ];

    /**
     * Create an EventAdapter instance for the given version string
     *
     * @param string $version Version string (e.g., '4.0.0.0')
     * @return EventAdapterInterface
     * @throws UnsupportedVersionException
     */
    public function create(string $version): EventAdapterInterface
    {
        if (isset(self::ADAPTER_CLASSES[$version])) {
            $adapterClass = self::ADAPTER_CLASSES[$version];
            return new $adapterClass();
        }

        $bestVersion = null;
        foreach (array_keys(self::ADAPTER_CLASSES) as $supportedVersion) {
            if (version_compare($supportedVersion, $version, '>')) {
                continue;
            }

            if ($bestVersion === null || version_compare($supportedVersion, $bestVersion, '>')) {
                $bestVersion = $supportedVersion;
            }
        }

        if ($bestVersion === null) {
            throw new UnsupportedVersionException(
                $version,
                array_keys(self::ADAPTER_CLASSES)
            );
        }

        $adapterClass = self::ADAPTER_CLASSES[$bestVersion];
        return new $adapterClass();
    }

    /**
     * @return string[]
     */
    public function getSupportedVersions(): array
    {
        return array_keys(self::ADAPTER_CLASSES);
    }

}
