<?php

declare(strict_types=1);

namespace Paysera\Delivery\Service;

use Paysera\Delivery\Exception\ConfigurationNotFoundException;

class ConfigLoader
{
    private string $configPath;

    public function __construct(string $configPath = null)
    {
        if ($configPath !== null) {
            $this->configPath = rtrim($configPath, '/') . '/';
        } else {
            $this->configPath = __DIR__ . '/../config/';
        }
    }

    /**
     * Load events configuration from file
     *
     * @param string $filename
     * @return array
     * @throws ConfigurationNotFoundException
     */
    public function loadEventsConfig(string $filename = 'events.php'): array
    {
        $filePath = $this->configPath . $filename;

        if (!file_exists($filePath)) {
            throw new ConfigurationNotFoundException(
                "Configuration file not found: {$filePath}"
            );
        }

        $config = require $filePath;

        if (!isset($config['events']) || !is_array($config['events'])) {
            throw new ConfigurationNotFoundException(
                "Invalid events configuration structure in {$filename}"
            );
        }

        return $config['events'];
    }

    /**
     * Load any configuration file
     *
     * @param string $filename
     * @return array
     * @throws ConfigurationNotFoundException
     */
    public function loadConfig(string $filename): array
    {
        $filePath = $this->configPath . $filename;

        if (!file_exists($filePath)) {
            throw new ConfigurationNotFoundException(
                "Configuration file not found: {$filePath}"
            );
        }

        $config = require $filePath;

        if (!is_array($config)) {
            throw new ConfigurationNotFoundException(
                "Configuration file must return an array: {$filename}"
            );
        }

        return $config;
    }

    /**
     * Set custom config path
     *
     * @param string $configPath
     * @return self
     */
    public function setConfigPath(string $configPath): self
    {
        $this->configPath = rtrim($configPath, '/') . '/';
        return $this;
    }

    /**
     * Get current config path
     *
     * @return string
     */
    public function getConfigPath(): string
    {
        return $this->configPath;
    }
}
