<?php

declare(strict_types=1);

namespace Paysera\Delivery\Service\CompatibilityService;

use Paysera\Delivery\Exception\UnsupportedVersionException;
use Paysera\Delivery\ValueObject\OpenCartVersion;

class RoutePathResolverFactory
{
    private const VERSION_SEPARATORS = [
        '4.0.0.0' => '|',
        '4.0.2.0' => '.',
    ];

    private const DEFAULT_SEPARATOR = '|';

    /**
     * @throws UnsupportedVersionException
     */
    public function create(OpenCartVersion $version, ?string $separator = null): RoutePathResolverInterface
    {
        $resolvedSeparator = $separator ?? $this->resolveSeparatorForVersion($version);

        return new RoutePathResolver($resolvedSeparator);
    }

    public function getSupportedVersions(): array
    {
        return array_map(
            fn(string $version) => new OpenCartVersion($version),
            array_keys(self::VERSION_SEPARATORS)
        );
    }

    public function isVersionSupported(OpenCartVersion $version): bool
    {
        return $this->findCompatibleVersion($version) !== null;
    }

    /**
     * @throws UnsupportedVersionException
     */
    private function resolveSeparatorForVersion(OpenCartVersion $version): string
    {
        $compatibleVersion = $this->findCompatibleVersion($version);

        if ($compatibleVersion === null) {
            throw new UnsupportedVersionException(
                $version->toString(),
                array_keys(self::VERSION_SEPARATORS)
            );
        }

        return self::VERSION_SEPARATORS[$compatibleVersion];
    }

    private function findCompatibleVersion(OpenCartVersion $requestedVersion): ?string
    {
        $availableVersions = array_keys(self::VERSION_SEPARATORS);
        $compatibleVersions = array_filter(
            $availableVersions,
            fn(string $version) => $requestedVersion->isGreaterThanOrEqual(new OpenCartVersion($version))
        );

        if (empty($compatibleVersions)) {
            return null;
        }

        usort($compatibleVersions, fn(string $a, string $b) => version_compare($b, $a));

        return $compatibleVersions[0];
    }
}
