<?php

declare(strict_types=1);

namespace Paysera\Delivery\Service\CompatibilityService;

use Exception;
use InvalidArgumentException;
use Paysera\Delivery\Exception\RoutePathResolutionException;

/**
 * Simple route path resolver with configurable separator
 *
 * Implements Single Responsibility Principle by focusing solely on route path resolution.
 * Uses Dependency Injection for separator configuration.
 */
class RoutePathResolver implements RoutePathResolverInterface
{
    private const BASE_ROUTE_PATH = 'extension/paysera_delivery/shipping/paysera';

    private string $separator;

    public function __construct(string $separator)
    {
        if (empty($separator)) {
            throw new InvalidArgumentException('Route separator cannot be empty');
        }

        $this->separator = $separator;
    }

    /**
     * @throws RoutePathResolutionException
     */
    public function getCompatibleRoutePath(string $action): string
    {
        if (empty($action)) {
            throw new RoutePathResolutionException($action, 'Action cannot be empty');
        }

        try {
            return $this->buildRoutePath($action);
        } catch (Exception $e) {
            throw new RoutePathResolutionException($action, 'Failed to resolve route path', $e);
        }
    }

    public function getRouteActionSeparator(): string
    {
        return $this->separator;
    }

    private function buildRoutePath(string $parts): string
    {
        return sprintf('%s%s%s', self::BASE_ROUTE_PATH, $this->separator, $parts);
    }
}
