<?php

declare(strict_types=1);

namespace Paysera\Delivery\Service\CompatibilityService\Polyfill\Version;

use Opencart\System\Library\Cart\Weight;
use Opencart\System\Library\Session;
use Opencart\System\Engine\Config;
use Opencart\System\Library\DB;
use Opencart\System\Engine\Proxy;
use Paysera\Delivery\Service\CompatibilityService\Polyfill\AbstractPolyfill;

/**
 * Polyfill for OpenCart version 4.1.0.0
 *
 * Introduces changes to the unit handling system while maintaining
 * compatibility with session and country retrieval methods.
 */
class Polyfill4_1_0_0 extends AbstractPolyfill
{
    public const VERSION = '4.1.0.0';

    public function getDefaultUnit(string $unitId, Weight $weight): string|int
    {
        return (int)$unitId;
    }

    /**
     * Session handling - same as 4.0.2.0 version
     */
    public function addShippingMethodLogo(Session $session): void
    {
        if (!isset($session->data['shipping_method'])) {
            return;
        }

        // Check if this is a Paysera shipping method
        $code = $session->data['shipping_method']['code'] ?? null;
        if ($code === null || !str_starts_with($code, 'paysera.')) {
            return;
        }

        // Check if nameWithLogo field exists before using it
        if (!isset($session->data['shipping_method']['nameWithLogo'])) {
            return;
        }

        $session->data['shipping_method']['name'] = $session->data['shipping_method']['nameWithLogo'];
    }

    /**
     * Session handling - same as 4.0.2.0 version
     */
    public function removeShippingMethodLogo(Session $session): void
    {
        if (!isset($session->data['shipping_method'])) {
            return;
        }

        // Check if this is a Paysera shipping method
        $code = $session->data['shipping_method']['code'] ?? null;
        if ($code === null || !str_starts_with($code, 'paysera.')) {
            return;
        }

        // Check if plainName field exists before using it
        if (!isset($session->data['shipping_method']['plainName'])) {
            return;
        }

        $session->data['shipping_method']['name'] = $session->data['shipping_method']['plainName'];
    }

    /**
     * Country retrieval - same as 4.0.2.0 version
     */
    public function getCountryByIsoCode2(
        string $iso_code_2,
        Config $config,
        DB $db,
        Proxy $modelLocalizationCountry
    ): ?array {
        $country = $modelLocalizationCountry->getCountryByIsoCode2($iso_code_2);

        return $country[0] ?? $country;
    }

    public function getVersion(): string
    {
        return self::VERSION;
    }
}
