<?php

declare(strict_types=1);

namespace Paysera\Delivery\Service\CompatibilityService\Polyfill;

use Opencart\System\Library\Session;
use Opencart\System\Library\Cart\Weight;
use Opencart\System\Engine\Config;
use Opencart\System\Library\DB;
use Opencart\System\Engine\Proxy;

/**
 * Interface PolyfillInterface
 *
 * Defines the contract for all polyfill version classes.
 * Each polyfill class should implement this interface to ensure consistency
 * and provide a common structure for version-specific compatibility fixes.
 */
interface PolyfillInterface
{
    /**
     * Get the OpenCart version this polyfill is designed for.
     *
     * @return string The version string (e.g., '4.0.0.0')
     */
    public function getVersion(): string;

    /**
     * Add shipping method logo based on version-specific logic.
     */
    public function addShippingMethodLogo(Session $session): void;

    /**
     * Remove shipping method logo based on version-specific logic.
     */
    public function removeShippingMethodLogo(Session $session): void;

    /**
     * Get country information by ISO code 2.
     *
     * @param string $iso_code_2 The ISO 2-letter country code
     * @param Config $config The OpenCart config object
     * @param DB $db The OpenCart database object
     * @param Proxy $modelLocalizationCountry The country model proxy
     * @return array|null The country data or null if not found
     */
    public function getCountryByIsoCode2(
        string $iso_code_2,
        Config $config,
        DB $db,
        Proxy $modelLocalizationCountry
    ): ?array;

    /**
     * Get the default unit for weight/cart operations.
     *
     * @param string $unitId The unit ID
     * @param Weight $weight The OpenCart weight object
     * @return string|int The default unit (string for older versions, int for newer)
     */
    public function getDefaultUnit(string $unitId, Weight $weight): string|int;
}
