<?php

declare(strict_types=1);

namespace Paysera\Delivery\Service\CompatibilityService\Polyfill;

use Paysera\Delivery\Service\CompatibilityService\Polyfill\Version\Polyfill4_0_0_0;
use Paysera\Delivery\Service\CompatibilityService\Polyfill\Version\Polyfill4_0_2_0;
use Paysera\Delivery\Service\CompatibilityService\Polyfill\Version\Polyfill4_1_0_0;

/**
 * Factory for creating polyfill instances based on version
 */
class PolyfillFactory
{
    /** @var array<string, class-string<PolyfillInterface>> */
    private const POLYFILL_CLASSES = [
        Polyfill4_0_0_0::VERSION => Polyfill4_0_0_0::class,
        Polyfill4_0_2_0::VERSION => Polyfill4_0_2_0::class,
        Polyfill4_1_0_0::VERSION => Polyfill4_1_0_0::class,
    ];

    /**
     * Create a polyfill instance for the given version string
     *
     * @param string $version Version string (e.g., '4.0.0.0')
     * @return PolyfillInterface
     * @throws \InvalidArgumentException If version is not supported
     */
    public function create(string $version): PolyfillInterface
    {
        if (!isset(self::POLYFILL_CLASSES[$version])) {
            throw new \InvalidArgumentException(
                sprintf('No polyfill found for version %s', $version)
            );
        }

        $polyfillClass = self::POLYFILL_CLASSES[$version];
        return new $polyfillClass();
    }

    /**
     * Get all supported versions
     *
     * @return string[]
     */
    public function getSupportedVersions(): array
    {
        return array_keys(self::POLYFILL_CLASSES);
    }

    /**
     * Check if a version is supported
     *
     * @param string $version
     * @return bool
     */
    public function isVersionSupported(string $version): bool
    {
        return isset(self::POLYFILL_CLASSES[$version]);
    }
}
