<?php

declare(strict_types=1);

namespace Paysera\Delivery\Service\CompatibilityService\Polyfill;

use Paysera\Delivery\ValueObject\OpenCartVersion as VersionValueObject;
use Opencart\System\Library\Session;
use Opencart\System\Library\Cart\Weight;
use Opencart\System\Engine\Config;
use Opencart\System\Library\DB;
use Opencart\System\Engine\Proxy;

class Polyfill
{
    private VersionValueObject $opencartVersion;
    private ?PolyfillInterface $activePolyfill = null;
    private PolyfillFactory $factory;

    public function __construct(VersionValueObject $opencartVersion, PolyfillFactory $factory)
    {
        $this->opencartVersion = $opencartVersion;
        $this->factory = $factory;
        $this->activePolyfill = $this->resolvePolyfill();
    }

    public function addShippingMethodLogo(Session $session): void
    {
        $this->activePolyfill?->addShippingMethodLogo($session);
    }

    public function removeShippingMethodLogo(Session $session): void
    {
        $this->activePolyfill?->removeShippingMethodLogo($session);
    }

    public function getCountryByIsoCode2(
        string $iso_code_2,
        Config $config,
        DB $db,
        Proxy $modelLocalizationCountry
    ): ?array {
        return $this->activePolyfill?->getCountryByIsoCode2($iso_code_2, $config, $db, $modelLocalizationCountry);
    }

    public function getDefaultUnit(string $unitId, Weight $weight): string|int
    {
        return $this->activePolyfill?->getDefaultUnit($unitId, $weight) ?? $unitId;
    }

    private function resolvePolyfill(): ?PolyfillInterface
    {
        $compatibleVersion = $this->findCompatibleVersion();

        if ($compatibleVersion === null) {
            return null;
        }

        return $this->factory->create($compatibleVersion);
    }

    private function findCompatibleVersion(): ?string
    {
        $availableVersions = $this->factory->getSupportedVersions();
        $compatibleVersions = array_filter($availableVersions, function (string $version): bool {
            return $this->opencartVersion->isGreaterThanOrEqual(new VersionValueObject($version));
        });

        if (empty($compatibleVersions)) {
            return null;
        }

        usort($compatibleVersions, function (string $a, string $b): int {
            return version_compare($b, $a);
        });

        return $compatibleVersions[0];
    }
}
