<?php

declare(strict_types=1);

namespace Paysera\Delivery\Service\CompatibilityService;

use Paysera\Delivery\Entity\Event;
use Paysera\Delivery\Service\Event\EventRegistrar;
use Paysera\Delivery\Exception\RoutePathResolutionException;
use Paysera\Delivery\ValueObject\OpenCartVersion;

class CompatibilityAdapter implements CompatibilityAdapterInterface
{
    private RoutePathResolverInterface $routePathResolver;
    private EventRegistrar $eventRegistrar;
    private OpenCartVersion $version;

    public function __construct(
        RoutePathResolverInterface $routePathResolver,
        EventRegistrar $eventRegistrar,
        OpenCartVersion $version
    ) {
        $this->routePathResolver = $routePathResolver;
        $this->eventRegistrar = $eventRegistrar;
        $this->version = $version;
    }

    public function getRoutePathResolver(): RoutePathResolverInterface
    {
        return $this->routePathResolver;
    }

    /**
     * @param Event[] $events
     * @throws RoutePathResolutionException
     */
    public function registerEvents(array $events): void
    {
        foreach ($events as $event) {
            $processedEvent = $this->processEvent($event);
            $this->eventRegistrar->registerEvent($processedEvent);
        }
    }

    /**
     * @param string[] $eventCodes
     */
    public function removeEvents(array $eventCodes): void
    {
        foreach ($eventCodes as $eventCode) {
            $this->eventRegistrar->removeEvent($eventCode);
        }
    }

    public function getVersion(): OpenCartVersion
    {
        return $this->version;
    }

    public function getEventRegistrar(): EventRegistrar
    {
        return $this->eventRegistrar;
    }

    /**
     * @throws RoutePathResolutionException
     */
    private function processEvent(Event $event): Event
    {
        $compatibleAction = $this->routePathResolver->getCompatibleRoutePath($event->getAction());

        return new Event(
            $event->getCode(),
            $event->getTrigger(),
            $compatibleAction,
            $event->getDescription(),
            $event->isStatus(),
            $event->getSortOrder()
        );
    }
}
