<?php

namespace Paysera\Delivery;

use Exception;
use RuntimeException;
use Paysera\Delivery\Entity\Order;
use Paysera\Delivery\Entity\Product;
use Paysera\Delivery\Entity\ShipmentRequest;
use Paysera\Delivery\Exception\RateLimitExceededException;
use Paysera\Delivery\Exception\ShipmentRequestGatewayNotFoundException;
use Paysera\Delivery\Service\Logger;
use Paysera\Delivery\Service\PartyCreator;
use Paysera\DeliveryApi\MerchantClient\Entity\OrderIdsList;
use Paysera\Scoped\Paysera\Component\RestClientCommon\Entity\Filter;
use Paysera\Scoped\Paysera\Component\RestClientCommon\Exception\ClientException;
use Paysera\Scoped\Paysera\Component\RestClientCommon\Exception\RequestException;
use Paysera\Delivery\Entity\DeliveryGatewaySettings;
use Paysera\DeliveryApi\MerchantClient\ClientFactory;
use Paysera\DeliveryApi\MerchantClient\Entity as Entities;
use Paysera\DeliveryApi\MerchantClient\Entity\CountryFilter;
use Paysera\DeliveryApi\MerchantClient\Entity\ShipmentPointCreate;
use Paysera\DeliveryApi\MerchantClient\MerchantClient;
use Paysera\Scoped\GuzzleHttp\Exception\RequestException as GuzzleRequestException;

class PayseraDeliveryLibraryHelper
{
    const BASE_URL = 'https://delivery-api.paysera.com/rest/v1/';
    const ORDER_PARTY_TYPE_RECEIVER = 'receiver';

    private $config;
    private $logger;
    private $session;
    private $weight;
    private $length;
    private $deliveryResolver;

    public function __construct($registry)
    {
        $this->config = $registry->get('config');
        $this->logger = new Logger($registry->get('log'));
        $this->session = $registry->get('session');
        $this->weight = $registry->get('weight');
        $this->length = $registry->get('length');
        $this->deliveryResolver = new PayseraDeliveryResolver();
    }

    /**
     * @param ?int $mac_id
     * @param ?string $mac_secret
     * @return ?MerchantClient
     */
    public function getMerchantClient($mac_id = null, $mac_secret = null)
    {
        $clientFactory = new ClientFactory([
            'base_url' => self::BASE_URL,
            'mac' => [
                'mac_id' => $mac_id
                    ?? (string) $this->config->get(PayseraDeliverySettings::PROJECT_ID),
                'mac_secret' => $mac_secret
                    ?? $this->config->get(PayseraDeliverySettings::PROJECT_PASSWORD),
            ],
            'timeout' => 10,
        ]);

        try {
            $merchantClient = $clientFactory->getMerchantClient();
        } catch (Exception $exception) {
            $this->logError($exception);

            return null;
        }

        return $merchantClient;
    }

    /**
     * @param ?int $mac_id
     * @param ?string $mac_secret
     * @return ?string
     */
    public function getProjectHash($mac_id = null, $mac_secret = null)
    {
        try {
            $projectHash = $this->getMerchantClient($mac_id, $mac_secret)
                ->getProjects(new Filter())
                ->getList()[0]
                ->getId()
            ;
        } catch (Exception $exception) {
            $this->logError($exception);

            return null;
        }

        return $projectHash;
    }

    /**
     * @return Entities\ShipmentGateway[]
     */
    public function getDeliveryGateways()
    {
        $merchantClient = $this->getMerchantClient();

        if ($merchantClient === null) {
            return [];
        }

        $gatewaysFilter = (new Entities\GatewaysFilter())
            ->setProjectId($this->config->get(PayseraDeliverySettings::PROJECT_HASH))
        ;

        try {
            $deliveryGateways = $merchantClient->updateGateway($gatewaysFilter)->getList();
        } catch (Exception $clientException) {
            $this->logError($clientException);

            return [];
        }

        return $deliveryGateways;
    }

    /**
     * @param string $code
     * @return ?Entities\ShipmentGateway
     */
    public function getDeliveryGatewayByCode($code)
    {
        $deliveryGateways = $this->getDeliveryGateways();

        foreach ($deliveryGateways as $deliveryGateway) {
            if ($deliveryGateway->getCode() === $code) {
                return $deliveryGateway;
            }
        }

        $this->logError(sprintf('Delivery Gateway code `%s` not found.', $code));

        return null;
    }

    /**
     * @return Entities\ShipmentMethod[]
     */
    public function getShippingMethods()
    {
        $merchantClient = $this->getMerchantClient();

        if ($merchantClient === null) {
            return [];
        }

        $methodsFilter = (new Entities\MethodsFilter())
            ->setProjectId($this->config->get(PayseraDeliverySettings::PROJECT_HASH))
        ;

        try {
            $shipmentMethods = $this->getMerchantClient()->updateMethod($methodsFilter)->getList();
        } catch (Exception $exception) {
            $this->logError($exception->getMessage());

            return [];
        }

        return $shipmentMethods;
    }

    /**
     * @param string $code
     * @return ?Entities\ShipmentMethod
     */
    public function getShippingMethodsByCode($code)
    {
        $shippingMethods = $this->getShippingMethods();

        foreach ($shippingMethods as $shippingMethod) {
            if ($shippingMethod->getCode() === $code) {
                return $shippingMethod;
            }
        }

        $this->logError(sprintf('Shipping method `%s` not found.', $code));

        return null;
    }

    /**
     * @param Entities\ShipmentGateway $shipmentGateway
     * @param string $countryCode
     * @param string $city
     * @return array
     */
    public function getTerminalByLocation($shipmentGateway, $countryCode, $city)
    {
        $merchantClient = $this->getMerchantClient();

        if ($merchantClient === null) {
            return [];
        }

        $parcelMachineFilter = (new Entities\ParcelMachineFilter())
            ->setShipmentGatewayCode($shipmentGateway->getCode())
            ->setCountry($countryCode)
            ->setCity($city)
        ;

        try {
            $parcelMachines = $merchantClient->getParcelMachines($parcelMachineFilter)->getList();
        } catch (Exception $exception) {
            $this->logError($exception->getMessage());

            return [];
        }

        $locations = [];

        foreach ($parcelMachines as $parcelMachine) {
            $locationName = trim(
                $parcelMachine->getAddress()->getStreet() . ', ' . $parcelMachine->getLocationName(),
                ', '
            );

            $locations[$parcelMachine->getId()] = $locationName;
        }

        asort($locations);

        return $locations;
    }

    /**
     * @param Entities\ShipmentGateway $shipmentGateway
     * @param string $countryCode
     * @return array
     */
    public function getCities($shipmentGateway, $countryCode)
    {
        $merchantClient = $this->getMerchantClient();

        if ($merchantClient === null) {
            return [];
        }

        $cityFilter = (new Entities\CityFilter())
            ->setCountry($countryCode)
            ->setGatewayCode($shipmentGateway->getCode())
        ;

        try {
            $cities = $merchantClient->getCities($cityFilter)->getItems();
        } catch (Exception $exception) {
            $this->logError($exception->getMessage());

            return [];
        }

        $normalizedCities = [];

        foreach ($cities as $city) {
            $normalizedCities[] = $city->getName();
        }

        asort($normalizedCities);

        return $normalizedCities;
    }

    /**
     * @param string
     * @return array
     */
    public function getCountries(string $deliveryGatewayCode)
    {
        $countryFilter = (new CountryFilter())->setShipmentGatewayCode($deliveryGatewayCode);

        $merchantClient = $this->getMerchantClient();

        if ($merchantClient === null) {
            return [];
        }

        try {
            $countries = $merchantClient->getCountries($countryFilter)->getItems();
        } catch (ClientException $clientException) {
            $this->logError($clientException);

            return [];
        }

        $normalizedCountries = [];

        foreach ($countries as $country) {
            $normalizedCountries[] = $country->getCountryCode();
        }

        asort($normalizedCountries);

        return $normalizedCountries;
    }

    /**
     * @param DeliveryGatewaySettings $deliveryGatewaySettings
     * @param Product[] $products
     * @param Order $order
     * @param ShipmentRequest $shipmentRequest
     *
     * @throws ShipmentRequestGatewayNotFoundException
     *
     * @return ?Entities\Order
     */
    public function createDeliveryOrder(
        DeliveryGatewaySettings $deliveryGatewaySettings,
        array $products,
        Order $order,
        ShipmentRequest $shipmentRequest
    ): ?Entities\Order {
        $merchantClient = $this->getMerchantClient();

        if ($merchantClient === null || $deliveryGatewaySettings === null) {
            return null;
        }

        $shipments = [];

        foreach ($products as $product) {
            for ($productQuantity = 1; $productQuantity <= $product->getQuantity(); $productQuantity++) {
                $shipments[] = (new Entities\ShipmentCreate())
                    ->setWeight(
                        $this->convertWeightToGrams((float)$product->getWeight(), (int)$product->getWeightClassId())
                    )
                    ->setLength(
                        $this->convertLengthToMillimeters((float)$product->getLength(), (int)$product->getLengthClassId())
                    )
                    ->setWidth(
                        $this->convertLengthToMillimeters((float)$product->getWidth(), (int)$product->getLengthClassId())
                    )
                    ->setHeight(
                        $this->convertLengthToMillimeters((float)$product->getHeight(), (int)$product->getLengthClassId())
                    )
                ;
            }
        }

        if ($deliveryGatewaySettings->getReceiverType() === PayseraDeliverySettings::TYPE_PARCEL_MACHINE) {
            $shipmentRequestGateway = $shipmentRequest->getShippingRequestGateway();
            if ($shipmentRequestGateway === null) {
                throw new ShipmentRequestGatewayNotFoundException();
            }

            $receiverAddress = (new Entities\Address())
                ->setCountry($shipmentRequestGateway->getIsoCode2())
                ->setState($shipmentRequestGateway->getZoneCode())
                ->setCity($shipmentRequestGateway->getCity())
            ;
        } else {
            $receiverAddress = (new Entities\Address())
                ->setCountry($order->getShippingIsoCode2())
                ->setState($order->getShippingZoneCode())
                ->setCity($order->getShippingCity())
                ->setStreet($order->getShippingAddress1())
                ->setPostalCode($order->getShippingPostcode())
            ;
        }

        $receiverParty = (new PartyCreator())
            ->createReceiverParty($order)
        ;

        $receiverContact = (new Entities\Contact())
            ->setParty($receiverParty)
            ->setAddress($receiverAddress)
        ;

        $orderCreate = (new Entities\OrderCreate())
            ->setProjectId($this->config->get(PayseraDeliverySettings::PROJECT_HASH))
            ->setShipmentGatewayCode($deliveryGatewaySettings->getShipmentGateway()->getCode())
            ->setShipmentMethodCode($this->deliveryResolver->resolveShippingMethodCode($deliveryGatewaySettings))
            ->setShipments($shipments)
            ->setReceiver(
                $this->createOrderParty(
                    $deliveryGatewaySettings->getReceiverType(),
                    self::ORDER_PARTY_TYPE_RECEIVER,
                    $receiverContact,
                    $shipmentRequest
                )
            )
            ->setEshopOrderId($order->getId())
        ;

        try {
            return $merchantClient->createOrder($orderCreate);
        } catch (RequestException $requestException) {
            $this->logError($requestException->getErrorDescription());
        } catch (Exception $exception) {
            $this->logError($exception);
        }

        return null;
    }

    /**
     * @param string $orderPartymethod
     * @param string $type
     * @param Entities\Contact $contact
     * @param ShipmentRequest $shipmentRequest
     *
     * @return ShipmentPointCreate
     */
    private function createOrderParty($orderPartymethod, $type, $contact, ShipmentRequest $shipmentRequest)
    {
        $orderParty = (new ShipmentPointCreate())
            ->setProjectId($this->config->get(PayseraDeliverySettings::PROJECT_HASH))
            ->setType($type)
            ->setSaved(false)
            ->setDefaultContact(false)
        ;

        if ($contact !== null) {
            $orderParty->setContact($contact);
        }

        if (
            $orderPartymethod === PayseraDeliverySettings::TYPE_PARCEL_MACHINE
            && $type === 'receiver'
        ) {
            $gateway = $shipmentRequest->getShippingRequestGateway();
            if ($gateway === null) {
                throw new ShipmentRequestGatewayNotFoundException();
            }

            $orderParty->setParcelMachineId($gateway->getTerminal());
        }

        return $orderParty;
    }

    /**
     * @param ShipmentRequest $shipmentRequest
     * @return bool
     *
     * @throws GuzzleRequestException
     */
    public function setShipmentStatusPrepaid(ShipmentRequest $shipmentRequest): bool
    {
        $merchantClient = $this->getMerchantClient();

        if ($merchantClient === null) {
            return false;
        }

        $merchantClient->createOrdersPrepaid(
            $this->getOrderIdsListFromShipmentRequest($shipmentRequest)
        );

        return true;
    }

    /**
     * @throws RateLimitExceededException
     * @throws ClientException
     * @throws RuntimeException
     * @throws ClientException
     */
    public function validateProjectCredentials($projectId, $password): bool
    {
        $credentials = (new Entities\ProjectCredentials())
            ->setProjectId($projectId)
            ->setPassword($password)
        ;

        try {
            $merchantClient = $this->getMerchantClient($projectId, $password);

            if ($merchantClient === null) {
                throw new RuntimeException('Failed to create merchant client during credentials validation');
            }

            return $merchantClient->validateProjectCredentials($credentials);
        } catch (ClientException $exception) {
            $response = $exception->getResponse();
            if ($response->getStatusCode() === 429) {
                throw new RateLimitExceededException(
                    'Rate limit exceeded. Too many validation attempts.',
                    429,
                    $exception
                );
            }

            $this->logError(sprintf('Project credentials validation failed: %s %s', $response->getStatusCode(), $response->getReasonPhrase()));

            throw $exception;
        }
    }

    private function getOrderIdsListFromShipmentRequest(ShipmentRequest $shipmentRequest): OrderIdsList
    {
        return new OrderIdsList([
            'order_ids' => [
                $shipmentRequest->getDeliveryOrderId(),
            ]
        ]);
    }

    private function logError(string $errorMessage): void
    {
        $this->logger->log(
            $errorMessage
        );
    }

    private function convertWeightToGrams(float $weight, int $weightClassId): float
    {
        $weightInDefaultUnits = $this->weight->convert(
            $weight,
            $weightClassId,
            PayseraDeliverySettings::WEIGHT_CLASS_FALLBACK_ID
        );

        return $weightInDefaultUnits * 1000;
    }

    private function convertLengthToMillimeters(float $length, int $lengthClassId): float
    {
        $lengthInDefaultUnits = $this->length->convert(
            $length,
            $lengthClassId,
            PayseraDeliverySettings::LENGTH_CLASS_FALLBACK_ID
        );

        return $lengthInDefaultUnits * 10;
    }
}
