<?php

declare(strict_types=1);

namespace Paysera\Delivery\Factory;

use Paysera\Delivery\Entity\Event;
use Paysera\Delivery\Exception\InvalidEventConfigurationException;

class EventFactory
{
    /**
     * @throws InvalidEventConfigurationException
     */
    public function createFromArray(array $config): Event
    {
        $this->validateConfig($config);

        return new Event(
            $config['code'],
            $config['trigger'],
            $config['action'],
            $config['description'] ?? '',
            $config['status'] ?? true,
            $config['sort_order'] ?? 1
        );
    }

    /**
     * @return Event[]
     * @throws InvalidEventConfigurationException
     */
    public function createFromArrayCollection(array $configs): array
    {
        $events = [];

        foreach ($configs as $index => $config) {
            try {
                $events[] = $this->createFromArray($config);
            } catch (InvalidEventConfigurationException $e) {
                throw new InvalidEventConfigurationException(
                    "Invalid event configuration at index {$index}: " . $e->getMessage()
                );
            }
        }

        return $events;
    }

    public function create(
        string $code,
        string $trigger,
        string $action,
        string $description = '',
        bool $status = true,
        int $sortOrder = 1
    ): Event {
        return new Event($code, $trigger, $action, $description, $status, $sortOrder);
    }

    /**
     * @throws InvalidEventConfigurationException
     */
    private function validateConfig(array $config): void
    {
        $requiredFields = ['code', 'trigger', 'action'];

        foreach ($requiredFields as $field) {
            if (!isset($config[$field]) || empty($config[$field])) {
                throw new InvalidEventConfigurationException(
                    "Missing or empty required field: {$field}"
                );
            }
        }

        if (!is_string($config['code'])) {
            throw new InvalidEventConfigurationException('Code must be a string');
        }

        if (!is_string($config['trigger'])) {
            throw new InvalidEventConfigurationException('Trigger must be a string');
        }

        if (!is_string($config['action'])) {
            throw new InvalidEventConfigurationException('Action must be a string');
        }

        if (isset($config['description']) && !is_string($config['description'])) {
            throw new InvalidEventConfigurationException('Description must be a string');
        }

        if (isset($config['status']) && !is_bool($config['status'])) {
            throw new InvalidEventConfigurationException('Status must be a boolean');
        }

        if (isset($config['sort_order']) && !is_int($config['sort_order'])) {
            throw new InvalidEventConfigurationException('Sort order must be an integer');
        }
    }
}
