<?php

namespace Opencart\Catalog\Model\Extension\PayseraDelivery\Shipping;

use Opencart\System\Engine\Model;
use Paysera\Delivery\DatabaseTable;
use Paysera\Delivery\DeliveryManager;
use Paysera\Delivery\Entity\DeliveryGatewaySettings;
use Paysera\Delivery\Entity\ShipmentRequest;
use Paysera\Delivery\Normalizer\DeliveryGatewaySettingsNormalizer;
use Paysera\Delivery\PayseraDeliveryLibraryHelper;
use Paysera\Delivery\PayseraDeliveryResolver;
use Paysera\Delivery\PayseraDeliverySettings;
use Paysera\Delivery\ShipmentGatewayManager;

if (is_file(DIR_EXTENSION . 'paysera_delivery/system/library/paysera/delivery/vendor/autoload.php')) {
    require_once DIR_EXTENSION . 'paysera_delivery/system/library/paysera/delivery/vendor/autoload.php';
}

class Paysera extends Model
{
    const CONFIG_PAYSERA_GEO = 'shipping_paysera_geo_zone_id';
    const ZERO_VAL = 0;

    private $delivery;
    private $deliveryManager;
    private $shipmentGatewayManager;
    private $deliveryGatewaySettingNormalizer;
    private $deliveryResolver;

    public function __construct($registry)
    {
        parent::__construct($registry);

        $this->delivery = new PayseraDeliveryLibraryHelper($this->registry);
        $this->deliveryManager = (new DeliveryManager($registry))->load();
        $this->shipmentGatewayManager = new ShipmentGatewayManager($registry, $this->delivery);
        $this->deliveryGatewaySettingNormalizer = new DeliveryGatewaySettingsNormalizer();
        $this->deliveryResolver = new PayseraDeliveryResolver();
    }

    public function getQuote($address)
    {
        $this->load->language('extension/paysera_delivery/shipping/paysera');

        $this->load->model('setting/setting');

        $deliveryGateways = $this->deliveryManager->getSettings()->getDeliveryGateways();

        $countryID = (int) $address['country_id'];
        $geoZoneID = (int) $this->config->get($this::CONFIG_PAYSERA_GEO);
        $zoneID = (int) $address['zone_id'];
        $zones = implode(',', [(int)$zoneID, (int)self::ZERO_VAL]);
        if (count($deliveryGateways) > 0) {

            $query = $this->db->query(
                'SELECT * FROM ' . DB_PREFIX . 'zone_to_geo_zone'
                . ' WHERE 1 '
                . ' AND geo_zone_id = ' . (int)$geoZoneID
                . ' AND country_id = ' . (int)$countryID
                . ' AND zone_id IN (' . $zones . ')'
                . ';'
            );

            if (!$this->config->get('shipping_flat_geo_zone_id')) {
                $status = true;
            } elseif ($query->num_rows) {
                $status = true;
            } else {
                $status = false;
            }
        } else {
            $status = false;
        }

        $method_data = [];

        if ($status === true) {
            $quote_data = [];
            $sort_order = [];

            $cartWeight = $this->cart->getWeight();
            foreach ($deliveryGateways as $deliveryGatewayCode => $shippingMethods) {
                foreach ($shippingMethods as $index => $shippingMethod) {
                    $normalizedShippingMethod = $this->deliveryGatewaySettingNormalizer->mapToEntity($shippingMethod);
                    $code = $this->deliveryResolver->resolveShippingMethodCode($normalizedShippingMethod);

                    if ($this->shipmentGatewayManager->isShippingMethodEnabled($code) === false) {
                        continue;
                    }

                    if (!$this->shipmentGatewayManager->isShippingMethodInGeoZone($countryID, $zones, $shippingMethod['geo_zone'])) {
                        continue;
                    }

                    if ($cartWeight < $shippingMethod['minimum_weight'] || $cartWeight > $shippingMethod['maximum_weight']) {
                        continue;
                    }

                    $fee = $this->isFreeDelivery($normalizedShippingMethod) ? 0 : $normalizedShippingMethod->getFee();
                    $taxClassId = $shippingMethod[PayseraDeliverySettings::TAX_CLASS_ID] ?? (int)$this->config->get('shipping_flat_tax_class_id') ?? 0;
                    $shippingMethodKey = sprintf('%s_%s_%d', $deliveryGatewayCode, $code, $index);
                    $quote_data[$shippingMethodKey] = [
                        'code' => PayseraDeliverySettings::SHIPPING_METHOD_PREFIX . '.' . $shippingMethodKey,
                        'title' => $normalizedShippingMethod->getTitle(),
                        'name' => $normalizedShippingMethod->getTitle(),
                        'plainName' => $normalizedShippingMethod->getTitle(),
                        'nameWithLogo' => sprintf(
                            '<img src="%s" class="paysera-delivery-gateway-logo"> %s',
                            $normalizedShippingMethod->getShipmentGateway()->getLogo(),
                            $normalizedShippingMethod->getTitle()
                        ),
                        'cost' => $fee,
                        'tax_class_id' => $taxClassId,
                        'text' => $this->currency->format(
                            $this->tax->calculate(
                                $fee,
                                $taxClassId,
                                $this->config->get('config_tax')
                            ),
                            $this->session->data['currency']
                        ),
                    ];

                    $sort_order[$shippingMethodKey] = $normalizedShippingMethod->getSortOrder() ?? 0;
                }
            }

            array_multisort($sort_order, SORT_ASC, $quote_data);

            if ($quote_data) {
                $method_data = [
                    'code' => PayseraDeliverySettings::SHIPPING_METHOD_PREFIX,
                    'name' => $this->language->get('text_title'),
                    'title' => $this->language->get('text_title'),
                    'quote' => $quote_data,
                    'sort_order' => $this->config->get('shipping_flat_sort_order'),
                    'error' => false,
                ];
            }
        }

        return $method_data;
    }

    /**
     * @param DeliveryGatewaySettings $shippingMethod
     * @return bool
     */
    private function isFreeDelivery($shippingMethod)
    {
        return $this->cart->getTotal() >= (int) $shippingMethod->getFreeDeliveryThreshold()
            && (int) $shippingMethod->getFreeDeliveryThreshold() !== 0
            ;
    }

    public function getGeoZone(int $geoZoneId)
    {
        $query = $this->db->query('SELECT * FROM ' . DB_PREFIX . 'zone_to_geo_zone WHERE geo_zone_id = \'' . (int)$geoZoneId . '\'');

        return $query->rows;
    }

    public function createShipmentRequest(ShipmentRequest $shipmentRequest): void
    {
        $gateway = $shipmentRequest->getShippingRequestGateway();

        $query = sprintf(
            <<<EOT
            INSERT INTO %s
                (order_id, delivery_order_id, shipping_method, status, gateway_terminal, gateway_iso_code_2, gateway_city, gateway_zone_code)
            VALUES ('%s', %s, '%s','%s', %s, %s, %s, %s);
            EOT,
            DatabaseTable::SHIPPING_REQUEST,
            $this->db->escape($shipmentRequest->getOrderId()),
            $this->encloseNullableString($shipmentRequest->getDeliveryOrderId()),
            $this->db->escape($shipmentRequest->getShippingMethod()),
            $this->db->escape($shipmentRequest->getStatus()),
            $this->encloseNullableString($gateway !== null ? $gateway->getTerminal() : null),
            $this->encloseNullableString($gateway !== null ? $gateway->getIsoCode2() : null),
            $this->encloseNullableString($gateway !== null ? $gateway->getCity() : null),
            $this->encloseNullableString($gateway !== null ? $gateway->getZoneCode() : null),
        );


        $this->db->query($query);
    }

    public function getShippingRequest(string $orderId): array
    {
        $query = sprintf(
            'SELECT * FROM %s WHERE order_id = \'%s\'',
            DatabaseTable::SHIPPING_REQUEST,
            $this->db->escape($orderId)
        );

        return $this->db->query($query)->row;
    }

    public function updateShipmentRequest(ShipmentRequest $shipmentRequest): void
    {
        $gateway = $shipmentRequest->getShippingRequestGateway();

        $query = sprintf(
            <<<EOT
            UPDATE %s
            SET shipping_method = '%s',
                status = '%s',
                gateway_terminal = %s,
                gateway_iso_code_2 = %s,
                gateway_city = %s,
                gateway_zone_code = %s,
                delivery_order_id = %s
            WHERE order_id = '%s';
            EOT,
            DatabaseTable::SHIPPING_REQUEST,
            $this->db->escape($shipmentRequest->getShippingMethod()),
            $this->db->escape($shipmentRequest->getStatus()),
            $this->encloseNullableString($gateway !== null ? $gateway->getTerminal() : null),
            $this->encloseNullableString($gateway !== null ? $gateway->getIsoCode2() : null),
            $this->encloseNullableString($gateway !== null ? $gateway->getCity() : null),
            $this->encloseNullableString($gateway !== null ? $gateway->getZoneCode() : null),
            $this->encloseNullableString($shipmentRequest->getDeliveryOrderId()),
            $this->db->escape($shipmentRequest->getOrderId())
        );

        $this->db->query($query);
    }

    private function encloseNullableString(?string $value): string
    {
        return $value !== null ? sprintf('\'%s\'', $this->db->escape($value)) : 'NULL';
    }
}
