<?php

/**
 * 2018 Paysera
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/afl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please email
 * to support@paysera.com, so we can send you a copy immediately.
 *
 * @author    Paysera <plugins@paysera.com>
 * @copyright 2018 Paysera
 * @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
 * International Registered Trademark & Property of Paysera
 */

declare(strict_types=1);

namespace Opencart\Catalog\Controller\Extension\OcPaysera\Payment;

require_once DIR_EXTENSION . 'oc_paysera/system/library/paysera/payment/vendor/autoload.php';

use Evp\Component\Money\Money;
use Exception;
use Opencart\System\Engine\Controller;
use Opencart\System\Engine\Registry;
use Paysera\CheckoutSdk\CheckoutFacade;
use Paysera\CheckoutSdk\CheckoutFacadeFactory;
use Paysera\CheckoutSdk\Entity\Request\PaymentCallbackValidationRequest;
use Paysera\CheckoutSdk\Entity\Request\PaymentRedirectRequest;
use Paysera\CheckoutSdk\Service\PaymentStatus;
use Paysera\Payment\Entity\Order;
use Paysera\Payment\Entity\PluginSettings;
use Paysera\Payment\Exception\OrderException;
use Paysera\Payment\Exception\PluginConfigurationException;
use Paysera\Payment\Helper\PaymentHelper;
use Paysera\Payment\Normalizer\PluginSettingsNormalizer;
use Paysera\Payment\PayseraPaymentSettings;
use Paysera\Payment\Service\PaymentMethodManager;
use Paysera\Payment\Service\PaymentMethodTranslator;
use Paysera\Payment\Service\PluginSettingsManager;
use Paysera\Payment\Service\Helper\AmountHelper;
use Paysera\Payment\Model\OrderModel;
use Paysera\Payment\Service\RoutePathResolver;
use Paysera\Payment\Service\SessionManager;
use Paysera\Payment\Normalizer\OrderNormalizer;
use Paysera\Payment\Normalizer\SessionNormalizer;
use Paysera\Payment\Service\Helper\BuyerConsentHelper;
use Paysera\Payment\Service\LanguageCodeReceiver;
use Paysera\Payment\Service\Logger;

class Paysera extends Controller
{
    const PAYSERA_PAYMENT_CONFIRMED = 'extension/oc_paysera/payment_callback/success';

    private const PLUGIN_VERSION = '0.8.3';
    private OrderModel $orderModel;
    private SessionManager $sessionManager;
    private Logger $logger;
    private PluginSettings $pluginSettings;
    private LanguageCodeReceiver $languageCodeReceiver;
    private CheckoutFacade $checkoutFacade;
    private RoutePathResolver $routePathResolver;
    private PaymentHelper $paymentHelper;

    public function __construct(Registry $registry)
    {
        parent::__construct($registry);

        $this->load->model('checkout/order');
        $this->load->model('checkout/shipping_method');
        $this->sessionManager = new SessionManager($this->session, new SessionNormalizer());
        $this->orderModel = new OrderModel(
            $this->model_checkout_order,
            new OrderNormalizer(),
            new AmountHelper($this->currency),
            $this->sessionManager,
            $this->db,
        );
        $this->pluginSettings = (new PluginSettingsManager($registry, new PluginSettingsNormalizer()))
            ->load()
            ->getSettings();
        $this->logger = new Logger($this->log);
        $this->languageCodeReceiver = new LanguageCodeReceiver($this->language);

        $this->checkoutFacade = (new CheckoutFacadeFactory)->create();

        $this->routePathResolver = new RoutePathResolver(VERSION);

        $this->paymentHelper = new PaymentHelper($this->logger);
    }

    public function index(): string
    {
        $this->load->language(PayseraPaymentSettings::EXTENSION);

        try {
            $this->checkProjectCredentials();
        } catch (PluginConfigurationException $exception) {
            return $exception->getMessage();
        }

        $order = $this->getOrderFromSession();

        $paymentMethodManager = new PaymentMethodManager($this->pluginSettings, $this->checkoutFacade);
        $paymentMethodManager->load($order);

        return $this->load->view(
            $this->getIndexTemplate(),
            [
                'action' => $this->url->link($this->routePathResolver->getCompatibleRoutePath('confirm')),
                'plugin_settings' => $this->pluginSettings,
                'payment_method_manager' => $paymentMethodManager,
                'payment_method_translator' => new PaymentMethodTranslator($this->languageCodeReceiver),
                'buyer_consent' => (new BuyerConsentHelper($this->language))->getBuyerConsent(),
            ]
        );
    }

    public function confirm(): void
    {
        error_reporting(E_ALL);
        ini_set('display_errors', '1');

        $_SERVER['HTTPS'] = $_SERVER['HTTPS'] ?? false;

        $this->checkProjectCredentials();
        $order = $this->getOrderFromSession();

        $redirectRequest = new PaymentRedirectRequest(
            $this->pluginSettings->getProjectId(),
            $this->pluginSettings->getProjectPassword(),
            $this->url->link($this->routePathResolver->getCompatibleRoutePath('accept')),
            $this->url->link($this->routePathResolver->getCompatibleRoutePath('cancel')),
            $this->url->link($this->routePathResolver->getCompatibleRoutePath('callback')),
            $order
        );

        $redirectRequest->setPayment($this->request->post['paysera_payment_method'] ?? null)
            ->setTest($this->pluginSettings->isTest())
            ->setBuyerConsent($this->pluginSettings->isBuyerConsent())
            ->setCountry($order->getPayerCountryCode())
            ->setLanguage($this->languageCodeReceiver->getPayseraLanguageCode())
            ->setPluginName('OpenCart')
            ->setPluginVersion(self::PLUGIN_VERSION)
            ->setCmsVersion(VERSION);

        $redirectResponse = $this->checkoutFacade->getPaymentRedirect($redirectRequest);

        $this->orderModel->addHistory($order->getOrderId(), $this->pluginSettings->getPendingStatus());

        $this->response->redirect($redirectResponse->getRedirectUrl());
    }

    public function cancel(): void
    {
        $this->sessionManager->clearCheckoutSessionParameters();
        $this->response->redirect($this->url->link('checkout/cart', [], true));
    }

    public function accept(): void
    {
        $this->checkProjectCredentials();

        $paymentValidationRequest = new PaymentCallbackValidationRequest(
            $this->pluginSettings->getProjectId(),
            $this->pluginSettings->getProjectPassword(),
            $this->request->get['data'] ?? ''
        );
        $paymentValidationRequest->setSs1($this->request->get['ss1'] ?? null)
            ->setSs2($this->request->get['ss2'] ?? null)
            ->setSs3($this->request->get['ss3'] ?? null);

        $response = $this->checkoutFacade->getPaymentCallbackValidatedData($paymentValidationRequest);

        $this->orderModel->addHistory($response->getOrder()->getOrderId(), $this->pluginSettings->getNewOrderStatus());
        $this->session->data['order_status_id'] = $this->pluginSettings->getNewOrderStatus();
        $this->cart->clear();

        $this->response->redirect($this->url->link('checkout/success', [], true));
    }

    public function callback(): void
    {
        try {
            $this->checkProjectCredentials();

            $paymentValidationRequest = new PaymentCallbackValidationRequest(
                $this->pluginSettings->getProjectId(),
                $this->pluginSettings->getProjectPassword(),
                $this->request->get['data'] ?? ''
            );
            $paymentValidationRequest->setSs1($this->request->get['ss1'] ?? null)
                ->setSs2($this->request->get['ss2'] ?? null)
                ->setSs3($this->request->get['ss3'] ?? null);

            $response = $this->checkoutFacade->getPaymentCallbackValidatedData($paymentValidationRequest);

            $order = $this->getOrder($response->getOrder()->getOrderId());

            if ($this->paymentHelper->isMerchantOrderPaid($response, $order)) {
                $this->load->model('setting/event');
                $this->event->trigger(
                    self::PAYSERA_PAYMENT_CONFIRMED,
                    [
                        $order->getOrderId(),
                    ]
                );
                $this->orderModel->addHistory($order->getOrderId(), $this->pluginSettings->getPaidStatus());
                exit('OK');
            }

            $orderExpected = ($order->getAmount() / 100)  . $order->getCurrency();
            $responseOrderReceived = ($response->getOrder()->getAmount() / 100) . $response->getOrder()->getCurrency();
            $responsePaymentReceived = (($response->getPaymentAmount() ?? 0) / 100) . $response->getPaymentCurrency();
            $this->logger->log(
                "Payment for order `{$order->getOrderId()}` is not valid."
                . " Expected: `$orderExpected` with status `" . PaymentStatus::SUCCESS . '`.'
                . " Received: order `$responseOrderReceived`"
                . " and payment `$responsePaymentReceived` with status `{$response->getStatus()}`."
            );
            exit("Payment for order `{$order->getOrderId()}` is not valid.");
        } catch (Exception $e) {
            $this->logger->log($e->getMessage());
            exit(get_class($e) . ': ' . $e->getMessage());
        }
    }

    public function header(&$route, &$data, &$output): void
    {
        if ($this->pluginSettings->isOwner()) {
            $data['analytics'][] = $this->load->view(
                PayseraPaymentSettings::EXTENSION . '_owner',
                [
                    'paysera_code' => $this->pluginSettings->getOwnerCode()
                ]
            );
        }
    }

    public function footer(&$route, &$data, &$output): void
    {
        if ($this->pluginSettings->isQuality()) {
            $this->checkProjectCredentials();

            $footer = $this->load->view(
                PayseraPaymentSettings::EXTENSION . '_quality',
                [
                    'paysera_project' => $this->pluginSettings->getProjectId(),
                    'paysera_lang' => $this->language->get('code'),
                ]
            );
            $output = $footer . $output;
        }
    }

    /**
     * @throws OrderException
     */
    private function getOrderFromSession(): Order
    {
        $sessionData = $this->sessionManager->getSessionData();

        return $this->getOrder($sessionData->getOrderId());
    }

    /**
     * @throws OrderException
     */
    private function getOrder(int $orderId): Order
    {
        $order = $this->orderModel->getOrder($orderId);

        if ($order === null) {
            $this->logger->log("Order with id `$orderId` not found.");
            throw new OrderException('Order not found.');
        }

        return $order;
    }

    /**
     * @throws PluginConfigurationException
     */
    private function checkProjectCredentials(): void
    {
        if ($this->pluginSettings->getProjectId() === null || $this->pluginSettings->getProjectPassword() === null) {
            throw new PluginConfigurationException('CONFIG ERROR: edit Paysera payment plugin configuration');
        }
    }

    private function getIndexTemplate(): string
    {
        $template = $this->config->get('config_template') . '/template/payment/paysera';
        return file_exists(DIR_TEMPLATE . $template) ? $template : PayseraPaymentSettings::EXTENSION;
    }
}
