<?php

/**
 * 2018 Paysera
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/afl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please email
 * to support@paysera.com, so we can send you a copy immediately.
 *
 * @author    Paysera <plugins@paysera.com>
 * @copyright 2018 Paysera
 * @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
 *  International Registered Trademark & Property of Paysera
 */

declare(strict_types=1);

namespace Opencart\Admin\Controller\Extension\OcPaysera\Payment;

require_once DIR_EXTENSION . 'oc_paysera/system/library/paysera/payment/vendor/autoload.php';

use Opencart\System\Engine\Controller;
use Opencart\System\Engine\Registry;
use Paysera\CheckoutSdk\CheckoutFacadeFactory;
use Paysera\Payment\Entity\Order;
use Paysera\Payment\Factory\OrderStatusRepositoryFactory;
use Paysera\Payment\Normalizer\PluginSettingsNormalizer;
use Paysera\Payment\Normalizer\SessionNormalizer;
use Paysera\Payment\PayseraPaymentSettings;
use Paysera\Payment\Repository\MessageRepository;
use Paysera\Payment\Service\LanguageCodeReceiver;
use Paysera\Payment\Service\PaymentMethodsReceiver;
use Paysera\Payment\Service\PaymentMethodTranslator;
use Paysera\Payment\Service\PluginSettingsManager;
use Paysera\Payment\Service\RoutePathResolver;
use Paysera\Payment\Service\SessionManager;
use Paysera\Payment\Validator\PluginSettingsValidator;

class Paysera extends Controller
{
    private const PAYSERA_EVENT_MENU_NAME = 'paysera_payment_menu';
    private const PAYSERA_EVENT_HEADER_NAME = 'paysera_header';
    private const PAYSERA_EVENT_FOOTER_NAME = 'paysera_footer';
    private const PAYSERA_SESSION_SAME_SITE = 'config_session_samesite';
    private const PAYSERA_CHECKOUT_ADDRESS = 'config_checkout_address';
    private const PAYSERA_EXTENSION_ABOUT_VIEW = 'extension/oc_paysera/payment/about';

    private const LANGUAGE_ABOUT = 'extension/oc_paysera/paysera/about';
    private const PAYSERA_EXTENSION_NO_DELIVERY_PLUGIN_VIEW = 'extension/oc_paysera/payment/no_delivery_plugin';
    private const LANGUAGE_NO_PLUGIN = 'extension/oc_paysera/paysera/no_delivery_plugin';
    private const PAYSERA_NO_DELIVERY_PLUGIN_ROUTE_ACTION = 'no_delivery_plugin';

    private SessionManager $sessionManager;
    private PluginSettingsManager $pluginSettingsManager;
    private RoutePathResolver $routePathResolver;
    private MessageRepository $messageRepository;

    public function __construct(Registry $registry)
    {
        parent::__construct($registry);

        $this->sessionManager = new SessionManager($this->session, new SessionNormalizer());
        $this->pluginSettingsManager = (new PluginSettingsManager($registry, new PluginSettingsNormalizer()))
            ->load();

        $this->routePathResolver = new RoutePathResolver(VERSION);

        $this->orderStatusRepositoryFactory = new OrderStatusRepositoryFactory($registry);
        $this->messageRepository = new MessageRepository($registry);
    }

    public function index(): void
    {
        $this->load->model('localisation/geo_zone');

        $this->document->setTitle($this->language->get('heading_title'));
        $this->document->addStyle(HTTP_CATALOG . 'extension/oc_paysera/admin/view/stylesheet/paysera/backoffice.css');
        $this->document->addScript(HTTP_CATALOG . 'extension/oc_paysera/admin/view/javascript/paysera/backoffice.js');

        $orderStatusRepository = $this->orderStatusRepositoryFactory->createInstance();

        $data = [
            'breadcrumbs' => $this->getBreadcrumbs(),
            'save' => $this->getUrlWithToken(PayseraPaymentSettings::EXTENSION . '|save'),
            'cancel' => $this->getUrlWithToken('marketplace/extension'),
            'order_statuses' => $orderStatusRepository->findAll(),
            'geo_zones' => $this->model_localisation_geo_zone->getGeoZones(),
            self::PAYSERA_SESSION_SAME_SITE => $this->config->get(self::PAYSERA_SESSION_SAME_SITE),
            self::PAYSERA_CHECKOUT_ADDRESS => $this->config->get(self::PAYSERA_CHECKOUT_ADDRESS),
        ];

        foreach ($this->pluginSettingsManager->getKeys() as $key) {
            $data[$key] = $this->request->post[$key] ?? $this->config->get($key);
            if ($key === 'payment_paysera_display_payments_list' && $data[$key] === '') {
                $data[$key] = true;
            }
        }

        $countries = $this->getPaymentCountries();
        if (!empty($countries)) {
            $data['paysera_countries'] = $countries;
            if (is_array($data['payment_paysera_category'])) {
                $data['paysera_selected_countries'] = [];
                foreach ($data['payment_paysera_category'] as $isoCode) {
                    $data['paysera_selected_countries'][$isoCode] = $data['paysera_countries'][$isoCode];
                }
            }
        }

        $this->responseOutput(PayseraPaymentSettings::EXTENSION, $data);
    }

    private function getBreadcrumbs(): array
    {
        $breadcrumbs = [
            'text_home' => 'common/dashboard',
            'text_extension' => 'marketplace/extension',
            'heading_title' => PayseraPaymentSettings::EXTENSION,
        ];

        $results = [];
        foreach ($breadcrumbs as $translationKey => $path) {
            $results[] = [
                'text' => $this->language->get($translationKey),
                'href' => $this->getUrlWithToken($path),
            ];
        }

        return $results;
    }

    public function about(): void
    {
        $this->messageRepository->loadLanguagePack(self::LANGUAGE_ABOUT);

        $this->document->setTitle($this->messageRepository->get('heading_title'));

        $this->responseOutput(self::PAYSERA_EXTENSION_ABOUT_VIEW, []);
    }

    public function no_delivery_plugin(): void
    {
        $this->messageRepository->loadLanguagePack(self::LANGUAGE_NO_PLUGIN);

        $this->document->setTitle($this->messageRepository->get('heading_title'));

        $this->responseOutput(self::PAYSERA_EXTENSION_NO_DELIVERY_PLUGIN_VIEW, []);
    }

    private function getUrlWithToken(string $path): string
    {
        $sessionData = $this->sessionManager->getSessionData();

        return $this->url->link(
            $path,
            [
                'user_token' => $sessionData->getUserToken(),
                'type' => 'payment',
            ]
        );
    }

    private function getPaymentCountries(): array
    {
        $paymentMethodsReceiver = new PaymentMethodsReceiver(
            $this->pluginSettingsManager->getSettings(),
            (new CheckoutFacadeFactory)->create()
        );
        $order = new Order(1, 100, 'EUR');
        $translator = new PaymentMethodTranslator(new LanguageCodeReceiver($this->language));

        $result = [];
        foreach ($paymentMethodsReceiver->getPaymentMethodCountries($order) as $country) {
            $result[$country->getCode()] = $translator->getTitle($country);
        }

        return $result;
    }

    public function payseraPaymentMenuHandler(string $eventRoute = '', array &$data = []): void
    {
        $payseraMenu = array_filter($data['menus'], function ($menu) {
            return $menu['id'] === 'menu-paysera';
        });

        if (!count($payseraMenu)) {
            $data['menus'][] = [
                'id' => 'menu-paysera',
                'name' => sprintf(
                    $this->messageRepository->get('menu_parent'),
                    '<img src="/extension/oc_paysera/admin/view/image/payment/paysera_menu_icon.svg" alt="paysera-logo" style="display: inline-block; margin-left: -6px; margin-right: 6px; width: 18px;"/>'
                ),
                'href' => '',
                'children' => [
                    'menu-paysera-about' => [
                        'name' => $this->messageRepository->get('menu_about'),
                        'href' => $this->getUrlWithToken($this->routePathResolver->getCompatibleRoutePath('about')),
                        'children' => [],
                    ],
                    'menu-paysera-delivery-settings' => [
                        'name' => $this->messageRepository->get('menu_delivery_settings'),
                        'href' => $this->getUrlWithToken(
                            $this->routePathResolver->getCompatibleRoutePath(self::PAYSERA_NO_DELIVERY_PLUGIN_ROUTE_ACTION)
                        ),
                        'children' => [],
                    ],
                    'menu-paysera-payment-settings' => [
                        'name' => $this->messageRepository->get('menu_payment_settings'),
                        'href' => $this->getUrlWithToken(PayseraPaymentSettings::EXTENSION),
                        'children' => [],
                    ],
                ],
            ];
        } else {
            $payseraMenuIndex = array_key_first($payseraMenu);
            $data['menus'][$payseraMenuIndex]['children']['menu-paysera-payment-settings'] = [
                'name' => $this->messageRepository->get('menu_payment_settings'),
                'href' => $this->getUrlWithToken(PayseraPaymentSettings::EXTENSION),
                'children' => [],
            ];
        }
    }

    public function install(): void
    {
        $this->load->model('setting/event');
        $this->load->model('setting/setting');
        $this->load->model(PayseraPaymentSettings::EXTENSION);

        $eventData = [
            [
                'code'        => self::PAYSERA_EVENT_MENU_NAME,
                'trigger'     => 'admin/view/common/column_left/before',
                'action'      => $this->routePathResolver->getCompatibleRoutePath('payseraPaymentMenuHandler'),
                'description' => '',
                'status'      => true,
                'sort_order'  => 1
            ],
            [
                'code' => self::PAYSERA_EVENT_HEADER_NAME,
                'description' => 'catalog/view/common/header/before',
                'action' => $this->routePathResolver->getCompatibleRoutePath('header'),
                'trigger' => 'catalog/view/common/header/before',
                'status' => 1,
                'sort_order' => 2,
            ],
            [
                'code' => self::PAYSERA_EVENT_FOOTER_NAME,
                'description' => 'catalog/view/common/footer/after',
                'action' => $this->routePathResolver->getCompatibleRoutePath('footer'),
                'trigger' => 'catalog/view/common/footer/after',
                'status' => 1,
                'sort_order' => 2,
            ],
        ];

        foreach ($eventData as $event) {
            if (version_compare(VERSION, '4.0.0.0', '==')) {
                $this->model_setting_event->addEvent(
                    $event['code'],
                    $event['description'],
                    $event['trigger'],
                    $event['action'],
                    $event['status'],
                    $event['sort_order']
                );
            } else {
                $this->model_setting_event->addEvent($event);
            }
        }

        $this->model_setting_setting->editValue('config', self::PAYSERA_SESSION_SAME_SITE, 'Lax');

        $this->model_setting_setting->editSetting(PluginSettingsManager::PLUGIN_SETTINGS_KEY, [
            'payment_paysera_new_order_status_id' => 2,
            'payment_paysera_paid_status_id' => 5,
            'payment_paysera_pending_status_id' => 1,
        ]);

        $enableCheckoutAddress = $this->model_extension_oc_paysera_payment_paysera->getSettingValue(
            'config',
            self::PAYSERA_CHECKOUT_ADDRESS
        );
        if ($enableCheckoutAddress === null) {
            $this->model_extension_oc_paysera_payment_paysera->addSettingValue(
                'config',
                self::PAYSERA_CHECKOUT_ADDRESS,
                '1'
            );
        } else {
            $this->model_setting_setting->editValue('config', self::PAYSERA_CHECKOUT_ADDRESS, '1');
        }
    }

    public function uninstall(): void
    {
        $this->load->model('setting/event');

        $this->model_setting_event->deleteEventByCode(self::PAYSERA_EVENT_MENU_NAME);
        $this->model_setting_event->deleteEventByCode(self::PAYSERA_EVENT_HEADER_NAME);
        $this->model_setting_event->deleteEventByCode(self::PAYSERA_EVENT_FOOTER_NAME);
    }

    public function save(): void
    {
        $this->load->language(PayseraPaymentSettings::EXTENSION);

        if (!$this->user->hasPermission('modify', PayseraPaymentSettings::EXTENSION)) {
            $json['error'] = $this->language->get('error_warning');
        } else {
            $pluginSettingsValidator = new PluginSettingsValidator(
                $this->orderStatusRepositoryFactory->createInstance(),
                new MessageRepository($this->registry)
            );

            $rules = [
                'payment_paysera_project' => 'required',
                'payment_paysera_sign' => 'required',
                'payment_paysera_new_order_status_id' => 'entity-exists',
                'payment_paysera_paid_status_id' => 'entity-exists',
                'payment_paysera_pending_status_id' => 'entity-exists',
            ];

            if (!$pluginSettingsValidator->validate($this->request->post, $rules)) {
                $json['error'] = $pluginSettingsValidator->getProcessedErrors();
            }
        }

        if (empty($json)) {
            $this->load->model('setting/setting');

            $this->model_setting_setting->editSetting(PluginSettingsManager::PLUGIN_SETTINGS_KEY, $this->request->post);
            $this->model_setting_setting->editValue(
                'config',
                self::PAYSERA_SESSION_SAME_SITE,
                $this->request->post[self::PAYSERA_SESSION_SAME_SITE]
            );
            $this->model_setting_setting->editValue(
                'config',
                self::PAYSERA_CHECKOUT_ADDRESS,
                $this->request->post[self::PAYSERA_CHECKOUT_ADDRESS]
            );
            $json['success'] = $this->language->get('text_success');
        }

        $this->response->addHeader('Content-Type: application/json');
        $this->response->setOutput(json_encode($json, JSON_THROW_ON_ERROR));
    }

    private function responseOutput(string $template, array $data): void
    {
        $data['header'] = $this->load->controller('common/header');
        $data['column_left'] = $this->load->controller('common/column_left');
        $data['footer'] = $this->load->controller('common/footer');

        $this->response->setOutput($this->load->view($template, $data));
    }
}
