<?php

declare(strict_types=1);

namespace Paysera\Payment\Normalizer;

use Paysera\Payment\Entity\Order;
use Paysera\Payment\Exception\OrderException;

class OrderNormalizer
{
    public function mapToEntity(array $orderData): Order
    {
        $orderAsArray = $this->filterEmptyStrings($orderData);

        if (!isset($orderAsArray['order_id'], $orderAsArray['amount_in_cents'], $orderAsArray['currency_code'])) {
            throw new OrderException('Unable to create order object. Required fields are missing.');
        }

        $order = new Order(
            (int) $orderAsArray['order_id'],
            (int) $orderAsArray['amount_in_cents'],
            (string) $orderAsArray['currency_code']
        );

        if (isset($orderAsArray['payment_firstname'])) {
            $order->setPayerFirstName((string) $orderAsArray['payment_firstname']);
        }
        if (isset($orderAsArray['payment_lastname'])) {
            $order->setPayerLastName((string) $orderAsArray['payment_lastname']);
        }
        if (isset($orderAsArray['email'])) {
            $order->setPayerEmail((string) $orderAsArray['email']);
        }
        if (isset($orderAsArray['payment_address_1']) || isset($orderAsArray['payment_address_2'])) {
            $payerAddress1 = $orderAsArray['payment_address_1'] ?? '';
            $payerAddress2 = $orderAsArray['payment_address_2'] ?? '';
            $payerStreet = trim($payerAddress1 . ' ' . $payerAddress2);
            if (!empty($payerStreet)) {
                $order->setPayerStreet($payerStreet);
            }
        }
        if (isset($orderAsArray['payment_city'])) {
            $order->setPayerCity((string) $orderAsArray['payment_city']);
        }
        if (isset($orderAsArray['payment_zone'])) {
            $order->setPayerState((string) $orderAsArray['payment_zone']);
        }
        if (isset($orderAsArray['payment_postcode'])) {
            $order->setPayerZip((string) $orderAsArray['payment_postcode']);
        }
        if (isset($orderAsArray['payment_iso_code_2'])) {
            $order->setPayerCountryCode((string) $orderAsArray['payment_iso_code_2']);
        }
        if (isset($orderAsArray['shipping_iso_code_2'])) {
            $order->setShippingCountryCode((string) $orderAsArray['shipping_iso_code_2']);
        }

        return $order;
    }

    public function filterEmptyStrings(array $data): array
    {
        return array_filter($data, static fn ($value) => $value !== '');
    }
}
