<?php

declare(strict_types=1);

namespace Paysera\Payment\Model;

use Opencart\Catalog\Model\Checkout\Order as OpencartOrderModel;
use Paysera\Payment\Entity\Order;
use Paysera\Payment\Entity\OrderShippingMethod;
use Paysera\Payment\Exception\OrderException;
use Paysera\Payment\Service\Helper\AmountHelper;
use Paysera\Payment\Normalizer\OrderNormalizer;
use Opencart\System\Engine\Proxy;
use Opencart\System\Library\Cart\Currency;
use Paysera\Payment\Service\SessionManager;

class OrderModel
{
    /**
     * ModelCheckoutOrder - catalog/model/checkout/order.php
     */
    private $orderModel;

    private OrderNormalizer $orderNormalizer;

    private AmountHelper $amountHelper;

    private SessionManager $sessionManager;

    /**
     * DB - system/library/db.php
     */
    private $db;

    public function __construct(
        $orderModel,
        OrderNormalizer $orderNormalizer,
        AmountHelper $amountHelper,
        SessionManager $sessionManager,
        $db
    ) {
        $this->orderModel = $orderModel;
        $this->orderNormalizer = $orderNormalizer;
        $this->amountHelper = $amountHelper;
        $this->sessionManager = $sessionManager;
        $this->db = $db;
    }

    public function getOrder(int $orderId): ?Order
    {
        $orderAsArray = $this->orderModel->getOrder($orderId);
        $this->actualizeOrder($orderId, $orderAsArray);

        if (!isset($orderAsArray['total'], $orderAsArray['currency_code'])) {
            throw new OrderException('Can not count order amount in cents. Necessary fields are absent.');
        }

        $orderAsArray['amount_in_cents'] = $this->amountHelper->getAmountInCents(
            (float) $orderAsArray['total'],
            (string) $orderAsArray['currency_code'],
        );

        return $this->orderNormalizer->mapToEntity($orderAsArray);
    }

    public function addHistory(int $orderId, int $orderStatusId): void
    {
        $this->orderModel->addHistory($orderId, $orderStatusId);
    }

    private function actualizeOrder(int $orderId, array &$orderData): void
    {
        $sessionData = $this->sessionManager->getSessionData();

        if (
            $sessionData->getShippingMethod() !== null
            && $orderData['shipping_method']['code'] !== $sessionData->getShippingMethod()->getCode()
        ) {
            $shippingMethod = $sessionData->getShippingMethod();
                $this->updateShippingDetails($orderId, $shippingMethod);
                $this->updateOrderTotal($orderId, $shippingMethod);
                $this->orderModel->addHistory(
                    $orderId,
                    $orderData['order_status_id'],
                    'Shipping method changed to ' . $shippingMethod->getTitle(),
                    false
                );

            $orderData = $this->orderModel->getOrder($orderId);
        }
    }

    private function updateShippingDetails($orderId, OrderShippingMethod $shippingMethod): void
    {
        $this->db->query(
            sprintf(
                <<<EOT
                    UPDATE `%sorder`
                        SET `shipping_method` = '%s'
                    WHERE `order_id` = %d
                EOT,
                DB_PREFIX,
                $this->db->escape($shippingMethod->getData()),
                $orderId
            )
        );
    }

    private function updateOrderTotal($orderId, OrderShippingMethod $shippingMethod): void
    {
        $this->db->query(
            sprintf(
                <<<EOT
                    UPDATE `%sorder_total`
                        SET `title` = '%s',
                            `value` = %f
                    WHERE `order_id` = %d AND `code` = 'shipping'
                EOT,
                DB_PREFIX,
                $this->db->escape($shippingMethod->getTitle()),
                $shippingMethod->getCost(),
                $orderId
            )
        );

        $queryTotal = $this->db->query(
            sprintf(
                <<<EOT
                    SELECT SUM(`value`) AS `total`
                    FROM `%sorder_total`
                    WHERE `order_id` = %d and `code` != 'total'
                EOT,
                DB_PREFIX,
                $orderId,
            )
        );

        $total = $queryTotal->rows[0] ?? [];

        $this->db->query(
            sprintf(
                <<<EOT
                    UPDATE `%sorder`
                        SET `total` = %f
                        WHERE `order_id` = %d
                EOT,
                DB_PREFIX,
                $total['total'] ?? 0,
                $orderId,
            )
        );

        $this->db->query(
            sprintf(
                <<<EOT
                    UPDATE `%sorder_total`
                        SET `value` = %f
                        WHERE `order_id` = %d and `code` = 'total'
                EOT,
                DB_PREFIX,
                $total['total'] ?? 0,
                $orderId,
            )
        );
    }
}
