<?php

namespace Paysera\Delivery;

use Exception;
use Paysera\Component\RestClientCommon\Entity\Filter;
use Paysera\Component\RestClientCommon\Exception\ClientException;
use Paysera\Component\RestClientCommon\Exception\RequestException;
use Paysera\Delivery\Entity\DeliveryGatewaySettings;
use Paysera\Delivery\Entity\Order;
use Paysera\Delivery\Entity\Product;
use Paysera\Delivery\Entity\ShipmentRequest;
use Paysera\Delivery\Exception\ShipmentRequestGatewayNotFoundException;
use Paysera\DeliveryApi\MerchantClient\ClientFactory;
use Paysera\DeliveryApi\MerchantClient\Entity as Entities;
use Paysera\DeliveryApi\MerchantClient\Entity\CountryFilter;
use Paysera\DeliveryApi\MerchantClient\Entity\OrderIdsList;
use Paysera\DeliveryApi\MerchantClient\Entity\ShipmentPointCreate;
use Paysera\DeliveryApi\MerchantClient\MerchantClient;

class PayseraDeliveryLibraryHelper
{
    const BASE_URL = 'https://delivery-api.paysera.com/rest/v1/';
    const PAYSERA_DELIVERY_EXCEPTION_TEXT = '[Paysera delivery] Got an exception: %s';
    const ORDER_PARTY_TYPE_RECEIVER = 'receiver';

    private $config;
    private $logger;
    private $weight;

    /**
     * Cart\Length - system/library/cart/length.php
     */
    private $lenght;
    private $deliveryResolver;

    public function __construct($registry)
    {
        $this->config = $registry->get('config');
        $this->logger = $registry->get('log');
        $this->weight = $registry->get('weight');
        $this->lenght = $registry->get('length');
        $this->deliveryResolver = new PayseraDeliveryResolver();
    }

    /**
     * @param ?int $mac_id
     * @param ?string $mac_secret
     * @return ?MerchantClient
     */
    public function getMerchantClient($mac_id = null, $mac_secret = null)
    {
        $clientFactory = new ClientFactory([
            'base_url' => self::BASE_URL,
            'mac' => [
                'mac_id' => $mac_id
                    ?? (string) $this->config->get(PayseraDeliverySettings::PROJECT_ID),
                'mac_secret' => $mac_secret
                    ?? $this->config->get(PayseraDeliverySettings::PROJECT_PASSWORD),
            ],
        ]);

        try {
            $merchantClient = $clientFactory->getMerchantClient();
        } catch (Exception $exception) {
            $this->logError($exception);

            return null;
        }

        return $merchantClient;
    }

    /**
     * @param ?int $mac_id
     * @param ?string $mac_secret
     * @return ?string
     */
    public function getProjectHash($mac_id = null, $mac_secret = null)
    {
        try {
            $projectHash = $this->getMerchantClient($mac_id, $mac_secret)
                ->getProjects(new Filter())
                ->getList()[0]
                ->getId()
            ;
        } catch (Exception $exception) {
            $this->logError($exception);

            return null;
        }

        return $projectHash;
    }

    /**
     * @return Entities\ShipmentGateway[]
     */
    public function getDeliveryGateways()
    {
        $merchantClient = $this->getMerchantClient();

        if ($merchantClient === null) {
            return [];
        }

        $gatewaysFilter = (new Entities\GatewaysFilter())
            ->setProjectId($this->config->get(PayseraDeliverySettings::PROJECT_HASH))
        ;

        try {
            $deliveryGateways = $merchantClient->updateGateway($gatewaysFilter)->getList();
        } catch (Exception $clientException) {
            $this->logError($clientException);

            return [];
        }

        return $deliveryGateways;
    }

    /**
     * @param string $code
     * @return ?Entities\ShipmentGateway
     */
    public function getDeliveryGatewayByCode($code)
    {
        $deliveryGateways = $this->getDeliveryGateways();

        foreach ($deliveryGateways as $deliveryGateway) {
            if ($deliveryGateway->getCode() === $code) {
                return $deliveryGateway;
            }
        }

        $this->logError(sprintf('Delivery Gateway code `%s` not found.', $code));

        return null;
    }

    /**
     * @return Entities\ShipmentMethod[]
     */
    public function getShippingMethods()
    {
        $merchantClient = $this->getMerchantClient();

        if ($merchantClient === null) {
            return [];
        }

        $methodsFilter = (new Entities\MethodsFilter())
            ->setProjectId($this->config->get(PayseraDeliverySettings::PROJECT_HASH))
        ;

        try {
            $shipmentMethods = $this->getMerchantClient()->updateMethod($methodsFilter)->getList();
        } catch (Exception $exception) {
            $this->logError($exception->getMessage());

            return [];
        }

        return $shipmentMethods;
    }

    /**
     * @param string $code
     * @return ?Entities\ShipmentMethod
     */
    public function getShippingMethodsByCode($code)
    {
        $shippingMethods = $this->getShippingMethods();

        foreach ($shippingMethods as $shippingMethod) {
            if ($shippingMethod->getCode() === $code) {
                return $shippingMethod;
            }
        }

        $this->logError(sprintf('Shipping method `%s` not found.', $code));

        return null;
    }

    /**
     * @param Entities\ShipmentGateway $shipmentGateway
     * @param string $countryCode
     * @param string $city
     * @return array
     */
    public function getTerminalByLocation($shipmentGateway, $countryCode, $city)
    {
        $merchantClient = $this->getMerchantClient();

        if ($merchantClient === null) {
            return [];
        }

        $parcelMachineFilter = (new Entities\ParcelMachineFilter())
            ->setShipmentGatewayCode($shipmentGateway->getCode())
            ->setCountry($countryCode)
            ->setCity($city)
        ;

        try {
            $parcelMachines = $merchantClient->getParcelMachines($parcelMachineFilter)->getList();
        } catch (Exception $exception) {
            $this->logError($exception->getMessage());

            return [];
        }

        $locations = [];

        foreach ($parcelMachines as $parcelMachine) {
            $locationName = trim(
                $parcelMachine->getAddress()->getStreet() . ', ' . $parcelMachine->getLocationName(),
                ', '
            );

            $locations[$parcelMachine->getId()] = $locationName;
        }

        asort($locations);

        return $locations;
    }

    /**
     * @param Entities\ShipmentGateway $shipmentGateway
     * @param string $countryCode
     * @return array
     */
    public function getCities($shipmentGateway, $countryCode)
    {
        $merchantClient = $this->getMerchantClient();

        if ($merchantClient === null) {
            return [];
        }

        $cityFilter = (new Entities\CityFilter())
            ->setCountry($countryCode)
            ->setGatewayCode($shipmentGateway->getCode())
        ;

        try {
            $cities = $merchantClient->getCities($cityFilter)->getItems();
        } catch (Exception $exception) {
            $this->logError($exception->getMessage());

            return [];
        }

        $normalizedCities = [];

        foreach ($cities as $city) {
            $normalizedCities[] = $city->getName();
        }

        asort($normalizedCities);

        return $normalizedCities;
    }

    /**
     * @param string
     * @return array
     */
    public function getCountries(string $deliveryGatewayCode)
    {
        $countryFilter = (new CountryFilter())->setShipmentGatewayCode($deliveryGatewayCode);

        $merchantClient = $this->getMerchantClient();

        if ($merchantClient === null) {
            return [];
        }

        try {
            $countries = $merchantClient->getCountries($countryFilter)->getItems();
        } catch (ClientException $clientException) {
            $this->logError($clientException);

            return [];
        }

        $normalizedCountries = [];

        foreach ($countries as $country) {
            $normalizedCountries[] = $country->getCountryCode();
        }

        asort($normalizedCountries);

        return $normalizedCountries;
    }

    /**
     * @param DeliveryGatewaySettings $deliveryGatewaySettings
     * @param Product[] $products
     * @param Order $order
     * @param ShipmentRequest $shipmentRequest
     *
     * @throws ShipmentRequestGatewayNotFoundException
     *
     * @return ?Entities\Order
     */
    public function createDeliveryOrder(
        DeliveryGatewaySettings $deliveryGatewaySettings,
        array $products,
        Order $order,
        ShipmentRequest $shipmentRequest
    ) {
        $merchantClient = $this->getMerchantClient();

        if ($merchantClient === null) {
            return null;
        }

        $shipments = [];

        foreach ($products as $product) {
            for ($productQuantity = 1; $productQuantity <= $product->getQuantity(); $productQuantity++) {
                $shipments[] = (new Entities\ShipmentCreate())
                    ->setWeight(
                        $this->weight->convert(
                            (int) $product->getWeight(),
                            $product->getWeightClassId(),
                            PayseraDeliverySettings::WEIGHT_CLASS_ID
                        )
                    )
                    ->setLength(
                        $this->lenght->convert(
                            (int) $product->getLength(),
                            $product->getLengthClassId(),
                            PayseraDeliverySettings::LENGTH_CLASS_ID
                        )
                    )
                    ->setWidth(
                        $this->lenght->convert(
                            (int) $product->getWidth(),
                            $product->getLengthClassId(),
                            PayseraDeliverySettings::LENGTH_CLASS_ID
                        )
                    )
                    ->setHeight(
                        $this->lenght->convert(
                            (int) $product->getHeight(),
                            $product->getLengthClassId(),
                            PayseraDeliverySettings::LENGTH_CLASS_ID
                        )
                    )
                ;
            }
        }

        $receiverParty = (new Entities\Party())
            ->setTitle($order->getPaymentFirstName() . ' ' . $order->getPaymentLastName())
            ->setEmail($order->getEmail())
            ->setPhone($order->getTelephone())
        ;

        if ($deliveryGatewaySettings->getReceiverType() === PayseraDeliverySettings::TYPE_PARCEL_MACHINE) {
            $shipmentRequestGateway = $shipmentRequest->getShippingRequestGateway();
            if ($shipmentRequestGateway === null) {
                throw new ShipmentRequestGatewayNotFoundException();
            }

            $receiverAddress = (new Entities\Address())
                ->setCountry($shipmentRequestGateway->getIsoCode2())
                ->setState($shipmentRequestGateway->getZoneCode())
                ->setCity($shipmentRequestGateway->getCity())
            ;
        } else {
            $receiverAddress = (new Entities\Address())
                ->setCountry($order->getShippingIsoCode2())
                ->setState($order->getShippingZoneCode())
                ->setCity($order->getShippingCity())
                ->setStreet($order->getShippingAddress1())
                ->setPostalCode($order->getShippingPostcode())
            ;
        }

        $receiverContact = (new Entities\Contact())->setParty($receiverParty)->setAddress($receiverAddress);

        $orderCreate = (new Entities\OrderCreate())
            ->setProjectId($this->config->get(PayseraDeliverySettings::PROJECT_HASH))
            ->setShipmentGatewayCode($deliveryGatewaySettings->getShipmentGateway()->getCode())
            ->setShipmentMethodCode($this->deliveryResolver->resolveShippingMethodCode($deliveryGatewaySettings))
            ->setShipments($shipments)
            ->setReceiver(
                $this->createOrderParty(
                    $deliveryGatewaySettings->getReceiverType(),
                    self::ORDER_PARTY_TYPE_RECEIVER,
                    $receiverContact,
                    $shipmentRequest
                )
            )
            ->setEshopOrderId($order->getId())
        ;

        try {
            return $merchantClient->createOrder($orderCreate);
        } catch (RequestException $requestException) {
            $this->logError($requestException->getErrorDescription());
        } catch (Exception $exception) {
            $this->logError($exception);
        }

        return null;
    }

    /**
     * @param ShipmentRequest $shipmentRequest
     * @return bool
     *
     * @throws GuzzleRequestException
     */
    public function setShipmentStatusPrepaid(ShipmentRequest $shipmentRequest): bool
    {
        $merchantClient = $this->getMerchantClient();

        if ($merchantClient === null) {
            return false;
        }

        $merchantClient->createOrdersPrepaid(
            $this->getOrderIdsListFromShipmentRequest($shipmentRequest)
        );

        return true;
    }

    private function getOrderIdsListFromShipmentRequest(ShipmentRequest $shipmentRequest): OrderIdsList
    {
        return new OrderIdsList([
            'order_ids' => [
                $shipmentRequest->getDeliveryOrderId(),
            ]
        ]);
    }

    /**
     * @param string $orderPartymethod
     * @param string $type
     * @param Entities\Contact $contact
     * @param ShipmentRequest $shipmentRequest
     *
     * @return ShipmentPointCreate
     */
    private function createOrderParty($orderPartymethod, $type, $contact, ShipmentRequest $shipmentRequest)
    {
        $orderParty = (new ShipmentPointCreate())
            ->setProjectId($this->config->get(PayseraDeliverySettings::PROJECT_HASH))
            ->setType($type)
            ->setSaved(false)
            ->setDefaultContact(false)
        ;

        if ($contact !== null) {
            $orderParty->setContact($contact);
        }

        if (
            $orderPartymethod === PayseraDeliverySettings::TYPE_PARCEL_MACHINE
            && $type === 'receiver'
        ) {
            $gateway = $shipmentRequest->getShippingRequestGateway();
            if ($gateway === null) {
                throw new ShipmentRequestGatewayNotFoundException();
            }

            $orderParty->setParcelMachineId($gateway->getTerminal());
        }

        return $orderParty;
    }

    /**
     * @param string $errorMessage
     */
    private function logError($errorMessage)
    {
        $this->logger->write(
            sprintf(self::PAYSERA_DELIVERY_EXCEPTION_TEXT, $errorMessage)
        );
    }
}
