<?php

namespace Paysera\Delivery;

use Paysera\Delivery\Entity\DeliveryGatewaySettings;
use Paysera\Delivery\Entity\DeliverySettings;
use Paysera\Delivery\Normalizer\DeliveryGatewaySettingsNormalizer;
use Paysera\Delivery\Normalizer\DeliverySettingsNormalizer;
use Paysera\DeliveryApi\MerchantClient\Entity\ShipmentGateway;
use Paysera\DeliveryApi\MerchantClient\Entity\ShipmentMethod;

class DeliveryManager
{
    private $deliverySettings;
    private $deliveryGatewaySettingsNormalizer;
    private $deliverySettingsNormalizer;
    private $deliveryResolver;
    private $registry;
    private $shipmentGatewayManager;

    public function __construct($registry)
    {
        $this->registry = $registry;
        $this->deliverySettingsNormalizer = new DeliverySettingsNormalizer();
        $this->deliveryGatewaySettingsNormalizer = new DeliveryGatewaySettingsNormalizer();
        $this->deliveryResolver = new PayseraDeliveryResolver();
        $this->shipmentGatewayManager = new ShipmentGatewayManager(
            $registry,
            new PayseraDeliveryLibraryHelper($registry)
        );
    }

    public function __get($key)
    {
        return $this->registry->get($key);
    }

    /**
     * @return $this
     */
    public function load()
    {
        $this->load->model('setting/setting');

        $this->deliverySettings = (new DeliverySettingsNormalizer())->mapToEntity(
            $this->model_setting_setting->getSetting(PayseraDeliverySettings::SETTINGS_NAME)
        );

        return $this;
    }

    /**
     * @return DeliverySettings
     */
    public function getSettings()
    {
        return $this->deliverySettings;
    }

    /**
     * @param string $code
     * @return $this
     */
    public function addGateway($code)
    {
        $deliveryGateways = $this->deliverySettings->getDeliveryGateways();
        $deliveryGateways[$code] = [];

        $this->deliverySettings->setDeliveryGateways($deliveryGateways);

        return $this;
    }

    /**
     * @param string $code
     * @return $this
     */
    public function removeGateway($code)
    {
        $deliveryGateways = $this->deliverySettings->getDeliveryGateways();
        unset($deliveryGateways[$code]);

        $this->deliverySettings->setDeliveryGateways($deliveryGateways);

        return $this;
    }

    /**
     * @param string $code
     * @param int $index
     * @return $this
     */
    public function removeGatewaySettings($code, $index)
    {
        $deliveryGateways = $this->getSettings()->getDeliveryGateways();

        array_splice($deliveryGateways[$code], $index, 1);

        $this->deliverySettings->setDeliveryGateways($deliveryGateways);

        return $this;
    }

    /**
     * @param string $code
     * @return DeliveryGatewaySettings[]
     */
    public function getGatewayByCode($code)
    {
        $deliveryGateways = $this->getSettings()->getDeliveryGateways();

        $normalizedDeliveryGateways = [];

        if (isset($deliveryGateways[$code]) && is_array($deliveryGateways[$code])) {
            foreach ($deliveryGateways[$code] as $deliveryGateway) {
                $normalizedDeliveryGateways[] = $this->deliveryGatewaySettingsNormalizer->mapToEntity($deliveryGateway);
            }
        }

        return $normalizedDeliveryGateways;
    }

    /**
     * @param string $code
     * @param int $index
     * @return DeliveryGatewaySettings
     */
    public function getGatewaySettings($code, $index)
    {
        $deliveryGateways = $this->getSettings()->getDeliveryGateways();

        if ($deliveryGateways === null) {
            return null;
        }

        if (isset($deliveryGateways[$code]) && isset($deliveryGateways[$code][$index])) {
            return $this->deliveryGatewaySettingsNormalizer->mapToEntity($deliveryGateways[$code][$index]);
        }

        return null;
    }

    /**
     * @param string $code
     * @param DeliveryGatewaySettings $deliveryGatewaySettings
     * @param ?int $index
     * @return ?DeliveryGatewaySettings
     */
    public function updateGatewaySettings($code, $deliveryGatewaySettings, $index = null)
    {
        $deliveryGateways = $this->getSettings()->getDeliveryGateways();

        if (!isset($deliveryGateways[$code]) || ($index !== null && !isset($deliveryGateways[$code][$index]))) {
            return null;
        }

        if ($index !== null) {
            $deliveryGateways[$code][$index] = $this->deliveryGatewaySettingsNormalizer
                ->mapFromEntity($deliveryGatewaySettings)
            ;
        } else {
            $deliveryGateways[$code][] = $this->deliveryGatewaySettingsNormalizer
                ->mapFromEntity($deliveryGatewaySettings)
            ;
        }

        $this->deliverySettings->setDeliveryGateways($deliveryGateways);

        $this->saveSettings();

        return $this->getGatewaySettings($code, $index ?? count($deliveryGateways[$code]) - 1);
    }

    /**
     * @param ShipmentGateway[] $deliveryGatewaySettings
     */
    public function syncDeliveryGateways($deliveryGatewaySettings)
    {
        foreach ($deliveryGatewaySettings as $deliveryGatewaySetting) {
            if (!$deliveryGatewaySetting->isEnabled()) {
                $this->removeGateway($deliveryGatewaySetting->getCode());
            }
        }

        $this->saveSettings();
    }

    /**
     * @param ShipmentMethod[] $shipmentMethodSettings
     */
    public function syncShipmentMethods($shipmentMethodSettings)
    {
        $deliveryGateways = $this->getSettings()->getDeliveryGateways();

        foreach ($deliveryGateways as $code => $deliveryGateway) {
            foreach ($deliveryGateway as $index => $deliveryGatewaySetting) {
                $shippingMethodCode = $this->deliveryResolver->resolveShippingMethodCode(
                    $this->deliveryGatewaySettingsNormalizer->mapToEntity($deliveryGatewaySetting)
                );

                if (!$this->shipmentGatewayManager->isShippingMethodEnabled($shippingMethodCode)) {
                    $this->removeGatewaySettings($code, $index);
                }
            }
        }

        $this->saveSettings();
    }

    /**
     * @param ?DeliverySettings $deliverySettings
     */
    public function saveSettings($deliverySettings = null)
    {
        $this->model_setting_setting->editSetting(
            PayseraDeliverySettings::SETTINGS_NAME,
            $this->deliverySettingsNormalizer->mapFromEntity($deliverySettings ?? $this->deliverySettings)
        );
    }

    /**
     * @param string $key
     * @param string $value
     */
    public function updateSettingValue($key, $value)
    {
        $this->model_setting_setting->editSettingValue(PayseraDeliverySettings::SETTINGS_NAME, $key, $value);
    }

    /**
     * @param string $code
     * @return bool
     */
    public function isGatewayCodeExists($code)
    {
        return array_key_exists($code, $this->getSettings()->getDeliveryGateways());
    }
}
