<?php

declare(strict_types=1);

namespace Paysera\Delivery\Validator;

use Paysera\DataValidator\Validator\AbstractValidator;
use Paysera\DataValidator\Validator\Exception\IncorrectValidationRuleStructure;
use Paysera\DataValidator\Validator\Rules\Comparison\GreaterThan;
use Paysera\DataValidator\Validator\Rules\Comparison\LessThan;
use Paysera\DataValidator\Validator\Rules\Min;
use Paysera\Delivery\Repository\MessageRepository;

class DeliveryGatewayValidator extends AbstractValidator
{
    protected MessageRepository $messageRepository;

    protected array $ruleSet = [
        'minimum_weight' => 'min:0|less-than:maximum_weight',
        'maximum_weight' => 'min:0|greater-than:minimum_weight',
        'fee' => 'min:0',
        'free_delivery_threshold' => 'min:0',
    ];

    /**
     * @throws IncorrectValidationRuleStructure
     */
    public function __construct(MessageRepository $messageRepository)
    {
        parent::__construct();

        $this->messageRepository = $messageRepository;

        $this->addRule(new Min());
        $this->addRule(new LessThan());
        $this->addRule(new GreaterThan());

        $this->setRuleMessage('min', $messageRepository->get('error_min_value'));
        $this->setRuleMessage('greater-than', $messageRepository->get('error_gt_value'));
        $this->setRuleMessage('less-than', $messageRepository->get('error_lt_value'));
    }

    public function validate($values, array $ruleSet = [], string $prefix = null): bool
    {
        if (empty($ruleSet)) {
            $ruleSet = $this->ruleSet;
        }

        return parent::validate($values, $ruleSet, $prefix);
    }

    public function getProcessedErrors(string $prefix = ''): array
    {
        if (!$this->hasErrors()) {
            return [];
        }

        $processedErrors = [
            $prefix . 'warning' => $this->messageRepository->get('form_has_errors'),
        ];
        foreach (parent::getProcessedErrors() as $field => $fieldErrors) {
            $processedErrors[$prefix . $field] = implode('<br>', $fieldErrors);
        }

        return $processedErrors;
    }
}
