<?php
declare(strict_types=1);

namespace Paysera\Delivery\Update;

use Paysera\Delivery\DatabaseTable;
use Exception;

class UpdateManager
{
    const INITIAL_UPDATE_VERSION = '1.1.2';

    private $config;
    private $db;
    private $modelSetting;
    private $modelSettingEvent;
    private $installedVersion;

    public function __construct($config, $db, $modelSetting, $modelSettingEvent)
    {
        $this->config = $config;
        $this->db = $db;
        $this->modelSetting = $modelSetting;
        $this->modelSettingEvent = $modelSettingEvent;
        $this->installedVersion = $this->config->get('paysera_delivery_version');
    }

    public function applyUpdates()
    {
        if ($this->isInstalled()) {
            $currentVersion = $this->installedVersion;
            $migrations = include __DIR__ . '/../config/migrations.php';

            if ($this->isUpdateNeeded($currentVersion, $migrations)) {
                foreach ($migrations as $version => $migrationClass) {
                    if (version_compare($currentVersion, $version, '<')) {
                        $this->applyMigration($migrationClass);
                    }
                }
            }
        }
    }

    private function isInstalled(): bool
    {
        if ($this->installedVersion !== null) {
            return true;
        }

        try {
            $this->db->query("SELECT 1 FROM `" . $this->db->escape(DatabaseTable::SHIPPING_REQUEST) . "` LIMIT 1");

            $this->updateVersion(self::INITIAL_UPDATE_VERSION);

            return true;
        } catch (Exception $e) {
            return false;
        }
    }

    private function isUpdateNeeded($currentVersion, $migrations): bool
    {
        $migrationVersions = array_keys($migrations);
        return version_compare($currentVersion, end($migrationVersions), '<');
    }

    private function applyMigration($migrationClass)
    {
        $migration = new $migrationClass($this->db, $this->modelSetting, $this->modelSettingEvent);
        $migration->apply();

        $this->updateVersion($migration->getVersion());
    }

    private function updateVersion($version)
    {
        $this->modelSetting->editSetting('paysera_delivery', [
            'paysera_delivery_version' => $version,
        ]);

        $this->installedVersion = $version;
    }
}
