<?php

declare(strict_types=1);

namespace Paysera\Magento2Paysera\Model;

use Magento\Checkout\Model\ConfigProviderInterface;
use Paysera\Magento2Paysera\Helper\Data;
use WebToPay;
use WebToPay_PaymentMethodCountry;
use WebToPayException;

class PayseraConfigProvider implements ConfigProviderInterface
{
    public const RESPONSE_STATUS_CONFIRMED = 1;
    public const RESPONSE_STATUS_ADDITIONAL_INFO = 3;
    public const PAYSERA_PAYMENT = 'payment/paysera';

    private const PROJECT_ID = 'projectid';
    private const EXTRA_CONFIG = 'paysera_extra';
    private const PAYMENT_TITLE = 'title';
    private const PAYMENT_DESCRIPTION = 'description';
    private const PAYMENT_ALL_COUNTRIES = 'allowspecific';
    private const PAYMENT_SELECTED_COUNTRIES = 'specificcountry';
    private const PAYMENT_GRID = 'grid';
    private const PAYMENT_BUYER_CONSENT = 'buyer_consent';
    private const PAYMENT_LIST = 'payment_list';
    private const DEFAULT_PROJECT_ID = 0;
    private const LINE_BREAK = '<div style="clear:both"><br/></div>';
    private const PAYSERA_LANGUAGES = [
        'lt',
        'lv',
        'et',
        'ru',
        'de',
        'pl',
        'en',
    ];
    private const DEFAULT_LANGUAGE = 'en';

    private $helper;

    public function __construct(Data $helper)
    {
        $this->helper = $helper;
    }

    public function getConfig(): array
    {
        return [
            'payment' => [
                'paysera' => [
                    'title' => __($this->helper->getPluginConfig(self::EXTRA_CONFIG, self::PAYMENT_TITLE)),
                    'countries' => $this->getPayseraPaymentMethods(),
                    'pageBaseUrl' => $this->helper->getPageBaseUrl(),
                ],
            ],
        ];
    }

    private function getStoreLanguage(): string
    {
        return in_array($this->helper->getStoreLanguageCode(), self::PAYSERA_LANGUAGES, true) ?
            $this->helper->getStoreLanguageCode() : self::DEFAULT_LANGUAGE
        ;
    }

    /**
     * @return WebToPay_PaymentMethodCountry[]
     * @throws WebToPayException
     */
    private function getPayseraCountries(): array
    {
        $projectId = $this->getValidProjectId($this->helper->getPluginConfig(self::PROJECT_ID));
        $currency = $this->helper->getOrderCurrencyCode();
        $amount = $this->helper->getTotalAmount();

        try {
            $countries = WebToPay::getPaymentMethodList($projectId, $amount, $currency)
                ->setDefaultLanguage($this->getStoreLanguage())
                ->getCountries()
            ;
        } catch (WebToPayException $exception) {
            return [];
        }

        return $countries;
    }

    private function getCountries(): array
    {
        $payseraCountries = $this->getPayseraCountries();

        $selectedCountries = $this->helper->getPluginConfig(self::EXTRA_CONFIG, self::PAYMENT_SELECTED_COUNTRIES);
        $selectedCountriesCodes = array_filter(
            explode(',', $selectedCountries ?? '')
        );

        $countries = [];
        foreach ($payseraCountries as $payseraCountry) {
            if (
                count($selectedCountriesCodes) > 0
                && in_array(strtoupper($payseraCountry->getCode()), $selectedCountriesCodes, true) === false
            ) {
                continue;
            }

            $countries[] = [
                'code' => $payseraCountry->getCode(),
                'title' => $payseraCountry->getTitle(),
                'groups' => $payseraCountry->getGroups(),
            ];
        }

        //TODO check is it works
        if (
            count($payseraCountries) === 0
            && in_array('other', array_column($countries, 'code'), true)
        ) {
            $otherCountry = $payseraCountries['other'];
            $countriesList[] = [
                'code' => $otherCountry->getCode(),
                'title' => $otherCountry->getTitle(),
                'groups' => $otherCountry->getGroups(),
            ];
        }

        return $countries;
    }

    private function getPayseraPaymentMethods(): string
    {
        $htmlBuilder = new BuildHtmlCode();

        $orderCountryCode = $this->helper->getOrderCountryCode();

        if ($this->helper->getPluginConfig(self::EXTRA_CONFIG, self::PAYMENT_LIST)) {
            $countries = $this->getCountries();

            if (count($countries) > 1) {
                $paymentMethodsHtml = $htmlBuilder->buildCountriesList($countries, $orderCountryCode);
                $paymentMethodsHtml .= self::LINE_BREAK;
            } else {
                $paymentMethodsHtml = '';
            }

            $paymentMethodsHtml .= $htmlBuilder->buildPaymentsList(
                $countries,
                $this->helper->getPluginConfig(self::EXTRA_CONFIG, self::PAYMENT_GRID),
                $orderCountryCode
            );
            $paymentMethodsHtml .= self::LINE_BREAK;
        } else {
            $description = $this->helper->getPluginConfig(self::EXTRA_CONFIG, self::PAYMENT_DESCRIPTION);

            if ($description !== null) {
                $paymentMethodsHtml = $description;
            } else {
                $paymentMethodsHtml = '';
            }
        }

        if ($this->helper->getPluginConfig(self::EXTRA_CONFIG, self::PAYMENT_BUYER_CONSENT)) {
            $paymentMethodsHtml .= self::LINE_BREAK;
            $paymentMethodsHtml .= $htmlBuilder->buildBuyerConsent();
        }

        return $paymentMethodsHtml;
    }

    private function getValidProjectId(string $projectId): int
    {
        if (filter_var($projectId, FILTER_VALIDATE_INT) !== false) {
            return (int) $projectId;
        }

        return self::DEFAULT_PROJECT_ID;
    }
}
