<?php

declare(strict_types=1);

namespace Paysera\Magento2Paysera\Service;

use Magento\Checkout\Model\Session;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Event\ManagerInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Quote\Model\QuoteFactory;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\Order;

class OrderService
{
    protected QuoteFactory $quoteFactory;
    protected CartRepositoryInterface $quoteRepository;
    protected OrderRepositoryInterface $orderRepository;
    protected SearchCriteriaBuilder $searchCriteriaBuilder;
    protected Session $session;
    protected ManagerInterface $eventManager;

    public function __construct(
        OrderRepositoryInterface $orderRepository,
        QuoteFactory $quoteFactory,
        CartRepositoryInterface $quoteRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        Session $session,
        ManagerInterface $eventManager
    ) {
        $this->orderRepository = $orderRepository;
        $this->quoteFactory = $quoteFactory;
        $this->quoteRepository = $quoteRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->session = $session;
        $this->eventManager = $eventManager;
    }

    public function getOrder(int $orderId): ?Order
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter('increment_id', $orderId, 'eq')
            ->create()
        ;

        $orders = $this->orderRepository->getList($searchCriteria)->getItems();

        return array_values($orders)[0] ?? null;
    }

    public function cancelOrder(Order $order): void
    {
        if ($order->getState() !== Order::STATE_CANCELED) {
            $order->cancel();
            $this->orderRepository->save($order);
        }
    }

    /**
     * @throws NoSuchEntityException
     */
    public function restoreCart(Order $order): void
    {
        $quote = $this->quoteRepository->get($order->getQuoteId());
        $quote
            ->setIsActive(true)
            ->setReservedOrderId(null)
        ;

        $this->quoteRepository->save($quote);
        $this->session->replaceQuote($quote);

        $this->eventManager
            ->dispatch(
                'restore_quote',
                ['order' => $order, 'quote' => $quote]
            )
        ;
    }
}
