<?php

namespace Paysera\Magento2Paysera\Controller\Index;

use Evp\Component\Money\Money;
use Magento\Framework\App\Action\Action;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\ResponseInterface;
use Magento\Framework\Controller\ResultInterface;
use Magento\Sales\Api\Data\OrderInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Framework\Exception\LocalizedException;
use Paysera\Magento2Paysera\Helper\Data as DataHelper;
use Paysera\Magento2Paysera\Model\PayseraConfigProvider;
use WebToPay;
use Exception;
use Magento\Framework\Controller\Result\Raw;
use Magento\Framework\Api\SearchCriteriaBuilder;

class Callback extends Action
{
    protected ScopeConfigInterface $scopeConfig;
    protected OrderRepositoryInterface $orderRepository;
    protected Raw $rawResult;
    protected SearchCriteriaBuilder $searchCriteriaBuilder;
    protected DataHelper $dataHelper;

    public function __construct(
        Context $context,
        ScopeConfigInterface $scopeConfig,
        OrderRepositoryInterface $orderRepository,
        Raw $rawResult,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        DataHelper $dataHelper
    ) {
        $this->scopeConfig = $scopeConfig;
        $this->orderRepository = $orderRepository;
        $this->rawResult = $rawResult;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->dataHelper = $dataHelper;

        parent::__construct($context);
    }

    /**
     * @return ResponseInterface|ResultInterface|void
     *
     * @throws Exception
     */
    public function execute()
    {
        $paysera_config = $this->scopeConfig->getValue(
            PayseraConfigProvider::PAYSERA_PAYMENT,
            ScopeInterface::SCOPE_STORE
        );

        $requestData = $this->dataHelper->getRequestData();

        try {
            $response = WebToPay::validateAndParseData(
                $requestData,
                $paysera_config['projectid'],
                $paysera_config['sign_password']
            );

            if ((int) $response['status'] === PayseraConfigProvider::RESPONSE_STATUS_CONFIRMED) {
                $searchCriteria = $this->searchCriteriaBuilder
                    ->addFilter('increment_id', $response['orderid'], 'eq')
                    ->create()
                ;

                $orders = $this->orderRepository->getList($searchCriteria)->getItems();

                if (count($orders) === 0) {
                    throw new Exception('Order not found.');
                }

                $order = array_values($orders)[0];

                if ($this->checkPayment($order, $response)) {
                    $order
                        ->setStatus($paysera_config['paysera_order_status']['order_status'])
                        ->save()
                    ;

                    return $this->rawResult->setContents('OK');
                }
            }
        } catch (Exception $e) {
            return $this->rawResult->setContents(get_class($e) . ': ' . $e->getMessage());
        }
    }

    /**
     * @param OrderInterface $order
     * @param array $response
     *
     * @return bool
     *
     * @throws LocalizedException
     */
    public function checkPayment(OrderInterface $order, array $response): bool
    {
        return $this->checkCurrency($order, $response)
            && $this->checkAmount($order, $response);
    }

    private function checkCurrency(OrderInterface $order, array $response): bool
    {
        $isNotSameToCurrency = $response['currency'] !== $order->getOrderCurrencyCode();
        $hasPaymentAmount = array_key_exists('payamount', $response);
        $isNotSameToPaymentCurrency = $hasPaymentAmount && $response['paycurrency'] !== $order->getOrderCurrencyCode();

        if ($isNotSameToCurrency || $isNotSameToPaymentCurrency) {
            throw new LocalizedException(__('Currencies do not match'));
        }

        return true;
    }

    private function checkAmount(OrderInterface $order, array $response): bool
    {
        $money = new Money($order->getGrandTotal());
        $responseAmount = Money::createFromNoDelimiterAmount($response['amount'], null);
        $responsePaymentAmount = array_key_exists('payamount', $response)
            ? Money::createFromNoDelimiterAmount($response['payamount'], null)
            : null;

        $isNotEqualToPaymentAmount = $responsePaymentAmount !== null && !$money->isEqual($responsePaymentAmount);

        if (!$money->isEqual($responseAmount) || $isNotEqualToPaymentAmount) {
            throw new LocalizedException(__('Amounts do not match'));
        }

        return true;
    }
}
