<?php
/**
 * 2018 Paysera
 *
 * NOTICE OF LICENSE
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 *  @author    Paysera <plugins@paysera.com>
 *  @copyright 2018 Paysera
 *  @license   GPL version 3 or later - http://www.gnu.org/licenses/gpl-3.0.html
 *  International Registered Trademark & Property of Paysera
 */

 defined('_JEXEC') or die('Restricted access');
 
/**
 * Build HTML code for Paysera Payments
 */
class PayseraHtmlForm
{
    /**
     * Code used for empty fields
     */
    const EMPTY_CODE     = '';

    /**
     * Selected field name
     */
    const FIELD_SELECTED = 'selected';

    /**
     * ID of other payments group
     */
    const CODE_OTHER     = 'other';

    /**
     * @return self
     */
    public static function create()
    {
        return new self();
    }

    /**
     * @param array  $countries
     * @param string $billingCountryCode
     *
     * @return string
     */
    public function buildCountriesList(
        $countries,
        $billingCountryCode
    ) {
        $defaultLangCode = $this->getDefaultLangCode(
            $countries,
            $billingCountryCode
        );

        $selectionField = '<select id="paysera_country" 
                                   class="payment-country-select" >';

        foreach ($countries as $country) {
            if ($country['code'] == $defaultLangCode) {
                $selected = $this::FIELD_SELECTED;
            } else {
                $selected = $this::EMPTY_CODE;
            }

            $selectionField .= '<option value="'
                . $country['code'] . '" '
                . $selected
                . '>';
            $selectionField .= $country['title'];
            $selectionField .= '</option>';
        }

        $selectionField     .= '</select>';

        return $selectionField;
    }

    /**
     * @param array   $countries
     * @param boolean $gridViewIsActive
     * @param string  $billingCountryCode
     *
     * @return string
     */
    public function buildPaymentsList(
        $countries,
        $gridViewIsActive,
        $billingCountryCode
    ) {
        $paymentsCode = $this::EMPTY_CODE;

        $defaultLangCode = $this->getDefaultLangCode(
            $countries,
            $billingCountryCode
        );

        foreach ($countries as $country) {
            $paymentsCode     .= '<div id="' . $country['code'] . '"';
            if ($gridViewIsActive) {
                $paymentsCode .= ' class="payment-countries paysera-payments grid"';
            } else {
                $paymentsCode .= ' class="payment-countries paysera-payments"';
            }
            $paymentsCode     .= ' style="display:';
            if (($country['code'] == $defaultLangCode)) {
                $paymentsCode .= 'block';
            } else {
                $paymentsCode .= 'none';
            }
            $paymentsCode     .= '">';

            $paymentsCode     .= $this->buildGroupList(
                $country['groups'],
                $country['code']
            );

            $paymentsCode     .= '</div>';
        }

        return $paymentsCode;
    }

    /**
     * @return string
     */
    public function buildBuyerConsent()
    {
        $translations = $this->getBuyerConsentTranslations();

        return sprintf(
            $translations['buyer_consent_text'],
            '<a href="' . $translations['buyer_consent_link'] . '"> ' . $translations['buyer_consent_rules'] .'</a>'
        );
    }

    /**
     * @return array
     */
    protected function getBuyerConsentTranslations()
    {
        $translations = array();

        if ($this->getTranslation('BUYER_CONSENT_TEXT') == "BUYER_CONSENT_TEXT") {
            $translations['buyer_consent_text'] = "Please be informed that the account information and payment initiation services will be provided to you by Paysera in accordance with these %s. By proceeding with this payment, you agree to receive this service and the service terms and conditions.";
        } else {
            $translations['buyer_consent_text'] = $this->getTranslation('BUYER_CONSENT_TEXT');
        }
        if ($this->getTranslation('BUYER_CONSENT_LINK') == "BUYER_CONSENT_LINK") {
            $translations['buyer_consent_link'] = "https://www.paysera.lt/v2/en-LT/legal/pis-rules-2020";
        } else {
            $translations['buyer_consent_link'] = $this->getTranslation('BUYER_CONSENT_LINK');
        }
        if ($this->getTranslation('BUYER_CONSENT_RULES') == "BUYER_CONSENT_RULES") {
            $translations['buyer_consent_rules'] = "rules";
        } else {
            $translations['buyer_consent_rules'] = $this->getTranslation('BUYER_CONSENT_RULES');
        }

        return $translations;
    }

    /**
     * @param string $transTag
     *
     * @return string
     */
    public function getTranslation($transTag)
    {
        return JText::_($transTag);
    }

    /**
     * @param array  $methods
     * @param string $countryCode
     *
     * @return string
     */
    protected function buildMethodsList(
        $methods,
        $countryCode
    ) {
        $paymentMethodCode = $this::EMPTY_CODE;
        foreach ($methods as $method) {
            $paymentMethodCode .= '<div id="' . $method->getKey() . '" class="payment">';

            $paymentMethodCode .= '<label>';
            $paymentMethodCode .= '<input class="rd_pay" ';
            $paymentMethodCode .= 'type="radio" ';
            $paymentMethodCode .= 'rel="r' . $countryCode . $method->getKey() . '" ';
            $paymentMethodCode .= 'name="payment[pay_type]" ';
            $paymentMethodCode .= 'value="' . $method->getKey() . '" /> ';

            $paymentMethodCode .= '<span class="paysera-text">';
            $paymentMethodCode .= $method->getTitle();
            $paymentMethodCode .= '</span>';

            $paymentMethodCode .= '<div class="paysera-image">';
            $paymentMethodCode .= '<img src="' . $method->getLogoUrl() . '" ';
            $paymentMethodCode .= 'alt="' . $method->getTitle() . '" />';
            $paymentMethodCode .= '</div>';

            $paymentMethodCode .= '</label>';
            $paymentMethodCode .= '</div>';
        }

        return $paymentMethodCode;
    }

    /**
     * @param array  $groups
     * @param string $countryCode
     *
     * @return string
     */
    protected function buildGroupList(
        $groups,
        $countryCode
    ) {
        $paymentGroupCode = $this::EMPTY_CODE;
        foreach ($groups as $group) {
            $paymentGroupCode .= '<div class="payment-group-wrapper">';

            $paymentGroupCode .= '<div class="payment-group-title">';
            $paymentGroupCode .= $group->getTitle();
            $paymentGroupCode .= '</div>';

            $paymentGroupCode .= $this->buildMethodsList(
                $group->getPaymentMethods(),
                $countryCode
            );
            $paymentGroupCode .= '</div>';
        }

        return $paymentGroupCode;
    }

    /**
     * @param array  $countries
     * @param string $countryCode
     *
     * @return string
     */
    protected function getDefaultLangCode(
        $countries,
        $countryCode
    ) {
        $countryCodes = array();

        foreach ($countries as $country) {
            $countryCodes[] = $country['code'];
        }

        if (in_array($countryCode, $countryCodes)) {
            $defaultLang = $countryCode;
        } elseif (in_array($this::CODE_OTHER, $countryCodes)) {
            $defaultLang = $this::CODE_OTHER;
        } else {
            $defaultLang = reset($countries)['code'];
        }

        return $defaultLang;
    }
}
