<?php
/**
 * 2018 Paysera
 *
 * NOTICE OF LICENSE
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 *  @author    Paysera <plugins@paysera.com>
 *  @copyright 2018 Paysera
 *  @license   GPL version 3 or later - http://www.gnu.org/licenses/gpl-3.0.html
 *  International Registered Trademark & Property of Paysera
 */

defined('_JEXEC') or die('Restricted access');

/**
 * Build Paysera payment methods list
 */
class PayseraAdminForm
{
    /**
     * Default value of string
     */
    const DEFAULT_TEXT = '';

    /**
     * Default value of int | bool
     */
    const DEFAULT_INT = 0;

    /**
     * Default lang code
     */
    const DEFAULT_LANG = 'en';

    /**
     * @var array
     */
    protected $tabs;

    /**
     * @var integer
     */
    protected $projectID;

    /**
     * @var string
     */
    protected $sign;

    /**
     * @var int
     */
    protected $test;

    /**
     * @var string
     */
    protected $lang;

    /**
     * @var string
     */
    protected $title;

    /**
     * @var string
     */
    protected $desc;

    /**
     * @var boolean
     */
    protected $paymentList;

    /**
     * @var boolean
     */
    protected $grid;

    /**
     * @var boolean
     */
    protected $buyerConsent;

    /**
     * @var array|string
     */
    protected $countriesSelected;

    /**
     * @var string
     */
    protected $defaultCountry;

    /**
     * @var string
     */
    protected $orderNew;

    /**
     * @var string
     */
    protected $orderPaid;

    /**
     * @var string
     */
    protected $orderPending;

    /**
     * @var boolean
     */
    protected $quality;

    /**
     * @var boolean
     */
    protected $ownership;

    /**
     * @var string
     */
    protected $ownershipCode;

    /**
     * @return self
     */
    public static function create()
    {
        return new self();
    }

    /**
     * PayseraAdminForm constructor
     */
    public function __construct()
    {
        $this->tabs           = array();
        $this->projectID      = $this::DEFAULT_TEXT;
        $this->sign           = $this::DEFAULT_TEXT;
        $this->test           = $this::DEFAULT_INT;
        $this->lang           = $this::DEFAULT_LANG;
        $this->title          = $this::DEFAULT_TEXT;
        $this->desc           = $this::DEFAULT_TEXT;
        $this->paymentList    = $this::DEFAULT_INT;
        $this->grid           = $this::DEFAULT_INT;
        $this->buyerConsent   = $this::DEFAULT_INT;
        $this->defaultCountry = $this::DEFAULT_TEXT;
        $this->orderNew       = $this::DEFAULT_INT;
        $this->orderPaid      = $this::DEFAULT_INT;
        $this->orderPending   = $this::DEFAULT_INT;
    }

    /**
     * @return array
     */
    public function getGeneralSettingTemplate()
    {
        $translations = $this->getGeneralSettingTranslations();

        $render = $this->renderGeneralSettingForm($translations);

        return $render;
    }

    /**
     * @return array
     */
    protected function getGeneralSettingTranslations()
    {
        $translations = array();

        if ($this->getTranslation('PAYSERA_GENERAL_PROJECT_ID') == "PAYSERA_GENERAL_PROJECT_ID") {
            $translations['project_id']['label'] = "Project ID";
        } else {
            $translations['project_id']['label'] = $this->getTranslation('PAYSERA_GENERAL_PROJECT_ID');
        }
        if ($this->getTranslation('PAYSERA_GENERAL_SIGN_PASS') == "PAYSERA_GENERAL_SIGN_PASS") {
            $translations['sign_pass']['label'] = "Sign";
        } else {
            $translations['sign_pass']['label'] = $this->getTranslation('PAYSERA_GENERAL_SIGN_PASS');
        }
        if ($this->getTranslation('PAYSERA_GENERAL_TEST_MODE') == "PAYSERA_GENERAL_TEST_MODE") {
            $translations['test']['label'] = "Test";
        } else {
            $translations['test']['label'] = $this->getTranslation('PAYSERA_GENERAL_TEST_MODE');
        }

        if ($this->getTranslation('PAYSERA_GENERAL_PROJECT_ID_DESC') == "PAYSERA_GENERAL_PROJECT_ID_DESC") {
            $translations['project_id']['desc'] = "Paysera project id";
        } else {
            $translations['project_id']['desc'] = $this->getTranslation('PAYSERA_GENERAL_PROJECT_ID_DESC');
        }
        if ($this->getTranslation('PAYSERA_GENERAL_SIGN_PASS_DESC') == "PAYSERA_GENERAL_SIGN_PASS_DESC") {
            $translations['sign_pass']['desc'] = "Paysera project password";
        } else {
            $translations['sign_pass']['desc'] = $this->getTranslation('PAYSERA_GENERAL_SIGN_PASS_DESC');
        }
        if ($this->getTranslation('PAYSERA_GENERAL_TEST_MODE_DESC') == "PAYSERA_GENERAL_TEST_MODE_DESC") {
            $translations['test']['desc'] = "Enable this to accept test payments";
        } else {
            $translations['test']['desc'] = $this->getTranslation('PAYSERA_GENERAL_TEST_MODE_DESC');
        }

        if ($this->getTranslation('BACKEND_GENERAL_SAVE') == "BACKEND_GENERAL_SAVE") {
            $translations['save'] = "Save main settings";
        } else {
            $translations['save'] = $this->getTranslation('BACKEND_GENERAL_SAVE');
        }

        return $translations;
    }

    /**
     * @param array $translations
     *
     * @return array
     */
    protected function renderGeneralSettingForm($translations)
    {
        return array(
            'fields_form' => $this->getGeneralSettingForm($translations),
            'settings' => $this->getGeneralSetting()
        );
    }

    /**
     * @param array $translations
     *
     * @return array
     */
    protected function getGeneralSettingForm($translations)
    {
        $generalForm = array();
        $generalForm[] = array(
            'form' => array(
                'input' => array(
                    $this->getNumberForm(
                        'PAYSERA_GENERAL_PROJECT_ID',
                        $translations['project_id'],
                        true
                    ),
                    $this->getTextForm(
                        'PAYSERA_GENERAL_SIGN_PASS',
                        $translations['sign_pass'],
                        true
                    ),
                    $this->getBoolForm(
                        'PAYSERA_GENERAL_TEST_MODE',
                        $translations['test']
                    )
                ),
                'submit' => array(
                    'title' => $translations['save']
                )
            )
        );

        return $generalForm;
    }

    /**
     * @return array
     */
    protected function getGeneralSetting()
    {
        $generalSetting = array();
        $generalSetting['PAYSERA_GENERAL_PROJECT_ID'] = $this->getProjectID();
        $generalSetting['PAYSERA_GENERAL_SIGN_PASS']  = $this->getSign();
        $generalSetting['PAYSERA_GENERAL_TEST_MODE']  = $this->getTest();

        return $generalSetting;
    }

    /**
     * @return array
     */
    public function getExtraSettingTemplate()
    {
        $translations = $this->getExtraSettingTranslations();

        $render = $this->renderExtraSettingForm($translations);

        return $render;
    }

    /**
     * @return array
     */
    protected function getExtraSettingTranslations()
    {
        $translations = array();

        if ($this->getTranslation('PAYSERA_EXTRA_TITLE') == "PAYSERA_EXTRA_TITLE") {
            $translations['title']['label'] = "Title";
        } else {
            $translations['title']['label'] = $this->getTranslation('PAYSERA_EXTRA_TITLE');
        }
        if ($this->getTranslation('PAYSERA_EXTRA_DESCRIPTION') == "PAYSERA_EXTRA_DESCRIPTION") {
            $translations['desc']['label'] = "Description";
        } else {
            $translations['desc']['label'] = $this->getTranslation('PAYSERA_EXTRA_DESCRIPTION');
        }
        if ($this->getTranslation('PAYSERA_EXTRA_LIST_OF_PAYMENTS') == "PAYSERA_EXTRA_LIST_OF_PAYMENTS") {
            $translations['list']['label'] = "List of payments";
        } else {
            $translations['list']['label'] = $this->getTranslation('PAYSERA_EXTRA_LIST_OF_PAYMENTS');
        }
        if ($this->getTranslation('PAYSERA_EXTRA_SPECIFIC_COUNTRIES') == "PAYSERA_EXTRA_SPECIFIC_COUNTRIES") {
            $translations['countries']['label'] = "Specific countries";
        } else {
            $translations['countries']['label'] = $this->getTranslation('PAYSERA_EXTRA_SPECIFIC_COUNTRIES');
        }
        if ($this->getTranslation('PAYSERA_EXTRA_GRIDVIEW') == "PAYSERA_EXTRA_GRIDVIEW") {
            $translations['grid']['label'] = "Grid view";
        } else {
            $translations['grid']['label'] = $this->getTranslation('PAYSERA_EXTRA_GRIDVIEW');
        }
        if ($this->getTranslation('PAYSERA_EXTRA_BUYER_CONSENT') == "PAYSERA_EXTRA_BUYER_CONSENT") {
            $translations['buyer_consent']['label'] = "Buyer consent";
        } else {
            $translations['buyer_consent']['label'] = $this->getTranslation('PAYSERA_EXTRA_BUYER_CONSENT');
        }
        if ($this->getTranslation('PAYSERA_EXTRA_FORCE_LOGIN') == "PAYSERA_EXTRA_FORCE_LOGIN") {
            $translations['force_login']['label'] = "Force Login";
        } else {
            $translations['force_login']['label'] = $this->getTranslation('PAYSERA_EXTRA_FORCE_LOGIN');
        }
        if ($this->getTranslation('PAYSERA_EXTRA_DEFAULT_COUNTRY') == "PAYSERA_EXTRA_DEFAULT_COUNTRY") {
            $translations['default_country']['label'] = "Default country";
        } else {
            $translations['default_country']['label'] = $this->getTranslation('PAYSERA_EXTRA_DEFAULT_COUNTRY');
        }

        if ($this->getTranslation('PAYSERA_EXTRA_TITLE_DESC') == "PAYSERA_EXTRA_TITLE_DESC") {
            $translations['title']['desc'] = "Payment method title that the customer will see on your website.";
        } else {
            $translations['title']['desc'] = $this->getTranslation('PAYSERA_EXTRA_TITLE_DESC');
        }
        if ($this->getTranslation('PAYSERA_EXTRA_DESCRIPTION_DESC') == "PAYSERA_EXTRA_DESCRIPTION_DESC") {
            $translations['desc']['desc'] = "This controls the description which the user sees during checkout.";
        } else {
            $translations['desc']['desc'] = $this->getTranslation('PAYSERA_EXTRA_DESCRIPTION_DESC');
        }
        if ($this->getTranslation('PAYSERA_EXTRA_LIST_OF_PAYMENTS_DESC') == "PAYSERA_EXTRA_LIST_OF_PAYMENTS_DESC") {
            $translations['list']['desc'] = "Enable this to display payment methods list at checkout page";
        } else {
            $translations['list']['desc'] = $this->getTranslation('PAYSERA_EXTRA_LIST_OF_PAYMENTS_DESC');
        }
        if ($this->getTranslation('PAYSERA_EXTRA_SPECIFIC_COUNTRIES_DESC') == "PAYSERA_EXTRA_SPECIFIC_COUNTRIES_DESC") {
            $translations['countries']['desc'] = "Select which country payment methods to display (empty means all)";
        } else {
            $translations['countries']['desc'] = $this->getTranslation('PAYSERA_EXTRA_SPECIFIC_COUNTRIES_DESC');
        }
        if ($this->getTranslation('PAYSERA_EXTRA_GRIDVIEW_DESC') == "PAYSERA_EXTRA_GRIDVIEW_DESC") {
            $translations['grid']['desc'] = "Enable this to use payments gridview";
        } else {
            $translations['grid']['desc'] = $this->getTranslation('PAYSERA_EXTRA_GRIDVIEW_DESC');
        }
        if ($this->getTranslation('PAYSERA_EXTRA_FORCE_LOGIN_DESC') == "PAYSERA_EXTRA_FORCE_LOGIN_DESC") {
            $translations['force_login']['desc'] = "Enable this to force customer to login on checkout";
        } else {
            $translations['force_login']['desc'] = $this->getTranslation('PAYSERA_EXTRA_FORCE_LOGIN_DESC');
        }
        if ($this->getTranslation('PAYSERA_EXTRA_DEFAULT_COUNTRY_DESC') == "PAYSERA_EXTRA_DEFAULT_COUNTRY_DESC") {
            $translations['default_country']['desc'] = "Select default country";
        } else {
            $translations['default_country']['desc'] = $this->getTranslation('PAYSERA_EXTRA_DEFAULT_COUNTRY_DESC');
        }

        if ($this->getTranslation('BACKEND_EXTRA_SAVE') == "BACKEND_EXTRA_SAVE") {
            $translations['save'] = "Save extra settings";
        } else {
            $translations['save'] = $this->getTranslation('BACKEND_EXTRA_SAVE');
        }

        return $translations;
    }

    /**
     * @param array $translations
     *
     * @return array
     */
    protected function renderExtraSettingForm($translations)
    {
        return array(
            'fields_form' => $this->getExtraSettingForm($translations),
            'settings' => $this->getExtraSetting()
        );
    }

    /**
     * @param array $translations
     *
     * @return array
     */
    protected function getExtraSettingForm($translations)
    {
        $extraForm = array();
        $extraForm[] = array(
            'form' => array(
                'input' => array(
                    $this->getBoolForm(
                        'PAYSERA_EXTRA_LIST_OF_PAYMENTS',
                        $translations['list']
                    ),
                    $this->getSelectForm(
                        'PAYSERA_EXTRA_SPECIFIC_COUNTRIES',
                        $translations['countries'],
                        $this->getCountries(),
                        false,
                        true
                    ),
                    $this->getBoolForm(
                        'PAYSERA_EXTRA_GRIDVIEW',
                        $translations['grid']
                    ),
                    $this->getBoolForm(
                        'PAYSERA_EXTRA_BUYER_CONSENT',
                        $translations['buyer_consent']
                    ),
                    $this->getSelectForm(
                        'PAYSERA_EXTRA_DEFAULT_COUNTRY',
                        $translations['default_country'],
                        $this->getCountries(),
                        false
                    ),
                ),
                'submit' => array(
                    'title' => $translations['save']
                )
            )
        );

        return $extraForm;
    }

    /**
     * @return array
     */
    protected function getExtraSetting()
    {
        $extraSetting = array();
        $extraSetting['PAYSERA_EXTRA_LIST_OF_PAYMENTS'] = $this->isPaymentList();
        $extraSetting['PAYSERA_EXTRA_GRIDVIEW'] = $this->isGrid();
        $extraSetting['PAYSERA_EXTRA_BUYER_CONSENT'] = $this->isBuyerConsent();
        $extraSetting['PAYSERA_EXTRA_SPECIFIC_COUNTRIES'] = $this->getCountriesSelected();
        $extraSetting['PAYSERA_EXTRA_DEFAULT_COUNTRY'] = $this->getDefaultCountry();

        return $extraSetting;
    }

    /**
     * @return array
     */
    public function getStatusSettingTemplate()
    {
        $translations = $this->getStatusSettingTranslations();

        $render = $this->renderStatusSettingForm($translations);

        return $render;
    }

    /**
     * @return array
     */
    protected function getStatusSettingTranslations()
    {
        $translations = array();

        if ($this->getTranslation('PAYSERA_ORDER_STATUS_NEW') == "PAYSERA_ORDER_STATUS_NEW") {
            $translations['new_order']['label'] = "New order status";
        } else {
            $translations['new_order']['label'] = $this->getTranslation('PAYSERA_ORDER_STATUS_NEW');
        }
        if ($this->getTranslation('PAYSERA_ORDER_STATUS_PAID') == "PAYSERA_ORDER_STATUS_PAID") {
            $translations['paid_order']['label'] = "Paid order status";
        } else {
            $translations['paid_order']['label'] = $this->getTranslation('PAYSERA_ORDER_STATUS_PAID');
        }
        if ($this->getTranslation('PAYSERA_ORDER_STATUS_PENDING') == "PAYSERA_ORDER_STATUS_PENDING") {
            $translations['pending_order']['label'] = "Pending order status";
        } else {
            $translations['pending_order']['label'] = $this->getTranslation('PAYSERA_ORDER_STATUS_PENDING');
        }

        if ($this->getTranslation('PAYSERA_ORDER_STATUS_NEW_DESC') == "PAYSERA_ORDER_STATUS_NEW_DESC") {
            $translations['new_order']['desc'] = "Order status after finishing checkout";
        } else {
            $translations['new_order']['desc'] = $this->getTranslation('PAYSERA_ORDER_STATUS_NEW_DESC');
        }
        if ($this->getTranslation('PAYSERA_ORDER_STATUS_PAID_DESC') == "PAYSERA_ORDER_STATUS_PAID_DESC") {
            $translations['paid_order']['desc'] = "Status for order with confirmed payment";
        } else {
            $translations['paid_order']['desc'] = $this->getTranslation('PAYSERA_ORDER_STATUS_PAID_DESC');
        }
        if ($this->getTranslation('PAYSERA_ORDER_STATUS_PENDING_DESC') == "PAYSERA_ORDER_STATUS_PENDING_DESC") {
            $translations['pending_order']['desc'] = "Order status for pending payment";
        } else {
            $translations['pending_order']['desc'] = $this->getTranslation('PAYSERA_ORDER_STATUS_PENDING_DESC');
        }

        if ($this->getTranslation('BACKEND_STATUS_SAVE') == "BACKEND_STATUS_SAVE") {
            $translations['save'] = "Save status settings";
        } else {
            $translations['save'] = $this->getTranslation('BACKEND_STATUS_SAVE');
        }

        return $translations;
    }

    /**
     * @param array $translations
     *
     * @return array
     */
    protected function renderStatusSettingForm($translations)
    {
        return array(
            'fields_form' => $this->getStatusSettingForm($translations),
            'settings' => $this->getStatusSetting()
        );
    }

    /**
     * @param array $translations
     *
     * @return array
     */
    protected function getStatusSettingForm($translations)
    {
        $statusForm = array();
        $statusForm[] = array(
            'form' => array(
                'input' => array(
                    $this->getStatusForm(
                        'PAYSERA_ORDER_STATUS_NEW',
                        $translations['new_order']
                    ),
                    $this->getStatusForm(
                        'PAYSERA_ORDER_STATUS_PAID',
                        $translations['paid_order']
                    ),
                    $this->getStatusForm(
                        'PAYSERA_ORDER_STATUS_PENDING',
                        $translations['pending_order']
                    ),
                ),
                'submit' => array(
                    'title' => $translations['save']
                )
            )
        );

        return $statusForm;
    }

    /**
     * @return array
     */
    protected function getStatusSetting()
    {
        $statusSetting = array();
        $statusSetting['PAYSERA_ORDER_STATUS_NEW'] = $this->getOrderNew();
        $statusSetting['PAYSERA_ORDER_STATUS_PAID'] = $this->getOrderPaid();
        $statusSetting['PAYSERA_ORDER_STATUS_PENDING'] = $this->getOrderPending();

        return $statusSetting;
    }

    /**
     * @return array
     */
    public function getAdditionsSettingTemplate()
    {
        $translations = $this->getAdditionsSettingTranslations();

        $render = $this->renderAdditionsSettingForm($translations);

        return $render;
    }

    /**
     * @return array
     */
    protected function getAdditionsSettingTranslations()
    {
        $translations = array();

        if ($this->getTranslation('PAYSERA_ADDITIONS_QUALITY_SIGN') == "PAYSERA_ADDITIONS_QUALITY_SIGN") {
            $translations['quality']['label'] = "Quality sign";
        } else {
            $translations['quality']['label'] = $this->getTranslation('PAYSERA_ADDITIONS_QUALITY_SIGN');
        }
        if ($this->getTranslation('PAYSERA_ADDITIONS_OWNERSHIP') == "PAYSERA_ADDITIONS_OWNERSHIP") {
            $translations['ownership']['label'] = "Ownership of website";
        } else {
            $translations['ownership']['label'] = $this->getTranslation('PAYSERA_ADDITIONS_OWNERSHIP');
        }
        if ($this->getTranslation('PAYSERA_ADDITIONS_OWNERSHIP_CODE') == "PAYSERA_ADDITIONS_OWNERSHIP_CODE") {
            $translations['ownership_code']['label'] = "Ownership code";
        } else {
            $translations['ownership_code']['label'] = $this->getTranslation('PAYSERA_ADDITIONS_OWNERSHIP_CODE');
        }

        if ($this->getTranslation('PAYSERA_ADDITIONS_QUALITY_SIGN_DESC') == "PAYSERA_ADDITIONS_QUALITY_SIGN_DESC") {
            $translations['quality']['desc'] = "Enable this to use quality sign";
        } else {
            $translations['quality']['desc'] = $this->getTranslation('PAYSERA_ADDITIONS_QUALITY_SIGN_DESC');
        }
        if ($this->getTranslation('PAYSERA_ADDITIONS_OWNERSHIP_DESC') == "PAYSERA_ADDITIONS_OWNERSHIP_DESC") {
            $translations['ownership']['desc'] = "Enable this to place ownership code";
        } else {
            $translations['ownership']['desc'] = $this->getTranslation('PAYSERA_ADDITIONS_OWNERSHIP_DESC');
        }
        if ($this->getTranslation('PAYSERA_ADDITIONS_OWNERSHIP_CODE_DESC') == "PAYSERA_ADDITIONS_OWNERSHIP_CODE_DESC") {
            $translations['ownership_code']['desc'] = "Enter ownership code";
        } else {
            $translations['ownership_code']['desc'] = $this->getTranslation('PAYSERA_ADDITIONS_OWNERSHIP_CODE_DESC');
        }
        if ($this->getTranslation('BACKEND_ADDITIONS_SAVE') == "BACKEND_ADDITIONS_SAVE") {
            $translations['save'] = "Save project additions settings";
        } else {
            $translations['save'] = $this->getTranslation('BACKEND_ADDITIONS_SAVE');
        }

        return $translations;
    }

    /**
     * @param array $translations
     *
     * @return array
     */
    protected function renderAdditionsSettingForm($translations)
    {
        return array(
            'fields_form' => $this->getAdditionsSettingForm($translations),
            'settings' => $this->getAdditionsSetting()
        );
    }

    /**
     * @param array $translations
     *
     * @return array
     */
    protected function getAdditionsSettingForm($translations)
    {
        $additionsForm = array();
        $additionsForm[] = array(
            'form' => array(
                'input' => array(
                    $this->getBoolForm(
                        'PAYSERA_ADDITIONS_QUALITY_SIGN',
                        $translations['quality']
                    ),
                    $this->getBoolForm(
                        'PAYSERA_ADDITIONS_OWNERSHIP',
                        $translations['ownership']
                    ),
                    $this->getTextForm(
                        'PAYSERA_ADDITIONS_OWNERSHIP_CODE',
                        $translations['ownership_code']
                    )
                ),
                'submit' => array(
                    'title' => $translations['save']
                )
            )
        );

        return $additionsForm;
    }

    /**
     * @return array
     */
    protected function getAdditionsSetting()
    {
        $additionsSetting = array();
        $additionsSetting['PAYSERA_ADDITIONS_QUALITY_SIGN'] = $this->isQuality();
        $additionsSetting['PAYSERA_ADDITIONS_OWNERSHIP'] = $this->isOwnership();
        $additionsSetting['PAYSERA_ADDITIONS_OWNERSHIP_CODE'] = $this->getOwnershipCode();

        return $additionsSetting;
    }

    /**
     * @return array
     */
    private function getCountries()
    {
        $countries = array();
        $projectID = $this->getProjectID();

        if (!$projectID) {
            return $countries;
        }

        try {
            $methods = WebToPay::getPaymentMethodList($projectID)
                ->setDefaultLanguage($this->getLang())
                ->getCountries()
            ;
        } catch (WebToPayException $exception) {
            return $countries;
        }

        foreach ($methods as $method) {
            $countries[] = array(
                'id' => $method->getCode(),
                'name' => $method->getTitle(),
            );
        }

        return $countries;
    }

    /**
     * @param string $key
     * @param array  $translations
     * @param bool   $requirement
     *
     * @return array
     */
    private function getTextForm($key, $translations, $requirement = false)
    {
        $textForm = array(
            'type'     => 'text',
            'label'    => $translations['label'],
            'name'     => $key,
            'required' => $requirement,
            'desc'     => $translations['desc']
        );

        return $textForm;
    }

    /**
     * @param string $key
     * @param array  $translations
     * @param bool   $requirement
     *
     * @return array
     */
    private function getNumberForm($key, $translations, $requirement = false)
    {
        $numberForm = array(
            'type'     => 'number',
            'label'    => $translations['label'],
            'name'     => $key,
            'required' => $requirement,
            'desc'     => $translations['desc']
        );

        return $numberForm;
    }

    /**
     * @param string $key
     * @param array  $translations
     * @param bool   $requirement
     *
     * @return array
     */
    private function getTextareaForm($key, $translations, $requirement = false)
    {
        $textareaForm = array(
            'type'     => 'textarea',
            'label'    => $translations['label'],
            'name'     => $key,
            'rows'     => 10,
            'required' => $requirement,
            'desc'     => $translations['desc'],
        );

        return $textareaForm;
    }

    /**
     * @param string $key
     * @param array  $translations
     * @param bool   $requirement
     *
     * @return array
     */
    private function getBoolForm($key, $translations, $requirement = false)
    {
        $textForm = array(
            'type'      => 'switch',
            'label'     => $translations['label'],
            'name'      => $key,
            'required'  => $requirement,
            'desc'      => $translations['desc'],
            'class'     => 'fixed-width-xxl',
            'is_bool'   => true,
            'values'    => array(
                array(
                    'id' => 'active_on',
                    'value' => 1
                ),
                array(
                    'id' => 'active_off',
                    'value' => 0
                )
            )
        );

        return $textForm;
    }

    /**
     * @param string $key
     * @param array $translations
     * @param array $selectList
     * @param bool $requirement
     * @param bool $multiple
     *
     * @return array
     */
    private function getSelectForm($key, $translations, $selectList, $requirement = false, $multiple = false)
    {

        $selectForm = array(
            'type'     => 'select',
            'label'    => $translations['label'],
            'name'     => $key,
            'required' => $requirement,
            'desc'     => $translations['desc'],
            'class'    => 'fixed-width-xxl',
            'multiple' => $multiple,
            'options'  => array(
                'query' => $selectList,
                'id'    => 'id',
                'name'  => 'name'
            )
        );

        if ($selectForm['multiple']) {
            $selectForm['size'] = 21;
        }

        return $selectForm;
    }

    /**
     * @param string $key
     * @param array $translations
     * @param bool $requirement
     *
     * @return array
     */
    private function getStatusForm($key, $translations, $requirement = false)
    {

        $statusForm = array(
            'type'     => 'status',
            'label'    => $translations['label'],
            'name'     => $key,
            'required' => $requirement,
            'desc'     => $translations['desc'],
        );

        return $statusForm;
    }

    /**
     * @param string $transTag
     *
     * @return string
     */
    public function getTranslation($transTag)
    {
        return JText::_($transTag);
    }

    /**
     * @param string $tabID
     * @param string $title
     * @param array  $content
     */
    public function addTab($tabID, $title, $content)
    {
        $oldTabsObj = $this->getTabs();
        $titleTranslation = $this->getTranslation($title);

        $oldTabsObj[] = array(
            'id'      => $tabID,
            'title'   => $titleTranslation,
            'content' => $content
        );

        $this->setTabs($oldTabsObj);
    }

    /**
     * @return array
     */
    public function getTabs()
    {
        return $this->tabs;
    }

    /**
     * @param array $tabs
     *
     * @return self
     */
    public function setTabs($tabs)
    {
        $this->tabs = $tabs;
        return $this;
    }

    /**
     * @return int
     */
    public function getProjectID()
    {
        return $this->projectID;
    }

    /**
     * @param int $projectID
     *
     * @return self
     */
    public function setProjectID($projectID)
    {
        $this->projectID = $projectID;
        return $this;
    }

    /**
     * @return string
     */
    public function getSign()
    {
        return $this->sign;
    }

    /**
     * @param string $sign
     *
     * @return self
     */
    public function setSign($sign)
    {
        $this->sign = $sign;
        return $this;
    }

    /**
     * @return int
     */
    public function getTest()
    {
        return $this->test;
    }

    /**
     * @param int $test
     *
     * @return self
     */
    public function setTest($test)
    {
        $this->test = $test;
        return $this;
    }

    /**
     * @return string
     */
    public function getLang()
    {
        return $this->lang;
    }

    /**
     * @param string $lang
     *
     * @return self
     */
    public function setLang($lang)
    {
        $this->lang = $lang;
        return $this;
    }

    /**
     * @return string
     */
    public function getTitle()
    {
        return $this->title;
    }

    /**
     * @param string $title
     *
     * @return self
     */
    public function setTitle($title)
    {
        $this->title = $title;
        return $this;
    }

    /**
     * @return string
     */
    public function getDesc()
    {
        return $this->desc;
    }

    /**
     * @param string $desc
     *
     * @return self
     */
    public function setDesc($desc)
    {
        $this->desc = $desc;
        return $this;
    }

    /**
     * @return bool
     */
    public function isPaymentList()
    {
        return $this->paymentList;
    }

    /**
     * @param bool $paymentList
     *
     * @return self
     */
    public function setPaymentList($paymentList)
    {
        $this->paymentList = $paymentList;
        return $this;
    }

    /**
     * @return bool
     */
    public function isGrid()
    {
        return $this->grid;
    }

    /**
     * @param bool $grid
     *
     * @return self
     */
    public function setGrid($grid)
    {
        $this->grid = $grid;
        return $this;
    }

    /**
     * @return bool
     */
    public function isBuyerConsent()
    {
        return $this->buyerConsent;
    }

    /**
     * @param bool $buyerConsent
     *
     * @return self
     */
    public function setBuyerConsent($buyerConsent)
    {
        $this->buyerConsent = $buyerConsent;
        return $this;
    }

    /**
     * @return array|string
     */
    public function getCountriesSelected()
    {
        return $this->countriesSelected;
    }

    /**
     * @param array|string $countriesSelected
     *
     * @return self
     */
    public function setCountriesSelected($countriesSelected)
    {
        $this->countriesSelected = $countriesSelected;
        return $this;
    }

    /**
     * @return string
     */
    public function getDefaultCountry()
    {
        return $this->defaultCountry;
    }

    /**
     * @param string $defaultCountry
     *
     * @return self
     */
    public function setDefaultCountry($defaultCountry)
    {
        $this->defaultCountry = $defaultCountry;
        return $this;
    }

    /**
     * @return string
     */
    public function getOrderNew()
    {
        return $this->orderNew;
    }

    /**
     * @param string $orderNew
     *
     * @return self
     */
    public function setOrderNew($orderNew)
    {
        $this->orderNew = $orderNew;
        return $this;
    }

    /**
     * @return string
     */
    public function getOrderPaid()
    {
        return $this->orderPaid;
    }

    /**
     * @param string $orderPaid
     *
     * @return self
     */
    public function setOrderPaid($orderPaid)
    {
        $this->orderPaid = $orderPaid;
        return $this;
    }

    /**
     * @return string
     */
    public function getOrderPending()
    {
        return $this->orderPending;
    }

    /**
     * @param string $orderPending
     *
     * @return self;
     */
    public function setOrderPending($orderPending)
    {
        $this->orderPending = $orderPending;
        return $this;
    }

    /**
     * @return bool
     */
    public function isQuality()
    {
        return $this->quality;
    }

    /**
     * @param bool $quality
     *
     * @return self
     */
    public function setQuality($quality)
    {
        $this->quality = $quality;
        return $this;
    }

    /**
     * @return bool
     */
    public function isOwnership()
    {
        return $this->ownership;
    }

    /**
     * @param bool $ownership
     *
     * @return self
     */
    public function setOwnership($ownership)
    {
        $this->ownership = $ownership;
        return $this;
    }

    /**
     * @return string
     */
    public function getOwnershipCode()
    {
        return $this->ownershipCode;
    }

    /**
     * @param string $ownershipCode
     *
     * @return self
     */
    public function setOwnershipCode($ownershipCode)
    {
        $this->ownershipCode = $ownershipCode;
        return $this;
    }
}
